;; Shift Register Service Routine
;;
;; -------------------------------------------------------------------------------------
;; FUNCTION:
;; CORE -> AY Chip SERIAL COMMUNICATION
;; This is the *MBHP CORE writes a byte data to the AY 3 8912 SOUND CHIP BOARD* ROUTINE 
;; -------------------------------------------------------------------------------------
;;
;; USES:
;; AY CHIP ADDRESS 	=> VAR_AY_CHIP_REG_ADDRESS  [8BIT BYTE, DEC. 0-16,  HEX 0x00-0x10]
;; AY CHIP DATA		=> VAR_AY_CHIP_REG_DATA	    [8BIT BYTE, DEC. 0-255, HEX 0x00-0xFF]
;;
;; 
;; ------------------------------[MEMO: 74HC595 - SHIFT REGISTER]-----------------------
;; DEFINITION of the PINS which are connected to the AY MODULE /
;; TTL 74HC595: 	8-BIT SERIAL-IN REGISTER WITH 8-BIT PARALLEL-OUTPUT LATCH
;; SERIAL CLOCK:   	SCLK (SCK ==> Shift Register Clock Input 		(PIN 11))
;; REGISTER CLOCK: 	RCLK (RCK ==> Storage Register Clock Input 		(PIN 12))
;; SERIAL DATA:    	SER  (SI  ==> Serial Data Input 				(PIN 14))
;; -------------------------------------------------------------------------------------

;; -----------[MEMO: BUS CONTROL SIGNALS OF THE AY CHIP (BDIR, BC1)]--------------------
;; -------------------------------------------------------------------------------------
;; MEMO AY 3 8912 CHIP BUS CONTROL [PIN CONNETION]
;; AY BDIR 	=> PIN 18
;; AY BC1 	=> PIN 20
;; NOTE: AY BC2	=> PIN 19 (BC2 ALWAYS => 1)
;;
;; 74HC595 - SHIFT REGISTER INTERCONNECTION
;; ASSIGNMENT => AY [8 BIT ADDRESS/DATA]
;;           -------------------------------------------------------------------------------------
;; BIT 	   	 [  7   |   6   |   5   |   4   |   3   |   2   |   1   |  0   ] BIT
;;           -------------------------------------------------------------------------------------
;; BIT 	  	 [A7/D7 | A6/D6 | A5/D5 | A4/D4 | A3/D3 | A2/D2 | A1/D1 | A0/D0] ADRESS / DATA
;;           -------------------------------------------------------------------------------------
;; 74HC595	 [Q0/P15| Q1/P01| Q2/P02| Q3/P03| Q4/P04| Q5/P05| Q6/P06|Q7/P07] PARALLEL OUTPUT / PIN
;;           -------------------------------------------------------------------------------------
;;
;; -----------------------------------------------------------------------------------------------
;; AY (YM) CHIP ADRESS & DATA WRITING:
;; FOR WRITING A BYTE TO THE AY (YM) CHIP IT TAKES 2 CYCLES 
;; CYCLE ONE: WRITE DATA BYTE ADDRESS 
;; CYCLE TWO: WRITE DATA BYTE
;;
;;
;; HOW TO WRITE A ADRESS AND A DATA BYTE TO THE AY CHIP:
;; [INACTIVE]->[OUTPUT ADDRESS]->[INACTIVE]->[OUTPUT DATA]->[INACTIVE]
;;
;; THE AY DATA-BUS CONTROL (BDIR, [BC2], BC1)
;; [BDIR]  -   [BC2]  -    [BC1]
;;   0		     1		     0	==> AY BUS INACTIVE	
;;   0		     1		     1	==> READ FROM THE AY CHIP
;;   1		     1  	     0	==> WRITE TO THE AY CHIP
;;   1		     1		     1	==> LATCH AY REGISTER ADDRESS
;;
;; WRITE ADRESS BYTE AND DATA BYTE TO THE AY CHIP (AY BUS CONTROL):
;; BUS CONTROL 	     [BDIR]     -     [BC1]
;; STEP1				1				1			==> WRITE ADDRESS
;; STEP2				0				0			==> INACTIVE (END CYCLE 1)
;; STEP3				1				0			==> WRITE DATA
;; STEP4				0				0			==> INACTIVE (END CYCLE 2)
;; 
;; 	
;; -----------[MEMO SET/CLEAR THE CONTROL SIGNALS]--------------------------------------
;; -------------------------------------------------------------------------------------
;; MEMO: CLEAR BDIR (AY Chip Control)
;; MEMO: bcf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	
;; 
;; MEMO: SET BDIR (AY Chip Control)
;; MEMO: bsf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	
;;
;; MEMO: CLEAR BC1 (AY Chip Control)
;; MEMO: bcf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	
;;
;; MEMO: SET BC1 (AY Chip Control)
;; MEMO: bsf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	
;;
;; ----------------[MEMO END]---------------------------------------------------------


 
; DEFINE PIC 18F PINS FOR SERIAL COMMUNICATION WITH THE 74HC595 SHIFT REGISTER
; ------------------------------------------------------------------------------------
AY_SR_LAT_SCLK				EQU		LATC	; SCLK 
AY_SR_PIN_SCLK				EQU		0 		; Pin15 RC0 CORE:J7 RC => 74HC595 PIN 11 [SCLK] 

AY_SR_LAT_RCLK				EQU		LATC	; RCLK
AY_SR_PIN_RCLK				EQU		1 		; Pin16 RC1 CORE:J7 SC => 74HC595 PIN 12 [RCLK]

AY_SR_LAT_SER_OUT			EQU		LATC	; SER
AY_SR_PIN_SER_OUT			EQU		2 		; Pin17 RC2 CORE:J7 SO => 74HC595 PIN 14 [SER]


; DEFINE PIC 18F PINS FOR AY CHIP BUS CONTROL / DIRECTION SIGNALS (BC1/BDIR)
; ---------------------------------------------------------------------------
AY_SR_LAT_BDIR	 			EQU		LATC	; BDIR
AY_SR_PIN_BDIR	  			EQU		4 		; Pin23 RC4 / CORE J10 RC [TO =>] TO AY BDIR
AY_SR_LAT_BC1	  			EQU		LATC	; BC1
AY_SR_PIN_BC1	  			EQU		5 		; Pin24 RC5 / CORE J10 SO [TO =>] TO AY BC1

AY_MIDI_NOTE_GATE_LAT		EQU		LATD	; MIDI Note On/Off Gate
AY_MIDI_GATE_PIN_RD4_J14	EQU		4		; MIDI NOTE ON/OFF GATE PIN RD4 MBHP Core8 J14



;; ***************************************************************************************
;; ---------------------------------------------------------------------------------------
;;           				AY_SR_WRITE_BIT MACRO
;; ---------------------------------------------------------------------------------------
;; ***************************************************************************************

;; -----------[MACRO: AY_SR_WRITE_BIT SERIAL DATA BIT BY BIT]-------
AY_SR_WRITE_BIT MACRO reg, bit
	;; MACRO FOR WRITING SERIAL DATA TO THE 74HC595 SHIFT REGISTER
	;; THE MACRO WRITES ONE BIT [AY ADRESS / AY DATA] TO SERIAL OUT
	
	;; SET PIN [SER] TO 0	
	bcf AY_SR_LAT_SER_OUT, AY_SR_PIN_SER_OUT
	
		btfsc reg, bit ; BIT TEST f, SKIP IF CLEAR
		;; reg => MIOS_PARAMETER1
		;; bit => BIT [0-7] OF MIOS_PARAMETER1 
		; SET SER PIN DEPENDING ON REGISTER CONTENT (reg,bit)
		; IS bit [0-7] in reg [MIOS_PARAMETER1] = 0 or 1 ?
		; CHECK IT WITH btfsc => BIT TEST f, SKIP IF CLEAR
			
			bsf AY_SR_LAT_SER_OUT, AY_SR_PIN_SER_OUT
			; GO HERE, IF THE BIT X in Y IS SET 
			; AND SET PIN [SER] TO 1
			; AND MAKE ANOTHER SCLK TRIGGER

			
			; IF THE BIT X IS *NOT* SET
			; PIN [SER] REMAIN 0 
		
			 
        ; ONE MORE SERIAL CLOCK TRIGGER => SCLK - 74HC595
        bsf     AY_SR_LAT_SCLK, AY_SR_PIN_SCLK	; rising clock edge
        bcf     AY_SR_LAT_SCLK, AY_SR_PIN_SCLK	; falling clock edge
		

	ENDM
	
;; **************************************************************************
;; --------------------------------------------------------------------------
;;           			END OF:	AY_SR_WRITE_BIT MACRO
;; --------------------------------------------------------------------------
;; **************************************************************************


;; **************************************************************************
;; --------------------------------------------------------------------------
;;             AY_SR_Write: WRITE A BYTE TO AN AY-3-8912 REGSITER
;; --------------------------------------------------------------------------
;; **************************************************************************
LABEL_AY_SR_Write
	;; FUNCTION:
	;; WRITING SERIAL Data (A ADRESS BYTE & A DATA BYTE) to the AY Chip
	;; 
	;; Note: THIS IS A time-critical ROUTINE
	;; SO DISABLE Interrupt Request (IRQ)
	IRQ_DISABLE

	;; MEMO: 
	;; AY CHIP BUS CONTROL
	;; BUS CONTROL 	      [BDIR]    -      [BC1]
	;; STEP1	             1	          	 1			==> WRITE ADDRESS 	(START WRITE ADR BYTE CYCLE)
	;; STEP2	             0	          	 0			==> INACTIVE 		(END ADR BYTE CYCLE)
	;; STEP3	             1	          	 0			==> WRITE DATA 		(START WRITE DATA BYTE CYCLE)
	;; STEP4	             0	          	 0			==> INACTIVE 		(END DATA BYTE CYCLE)
  
	;; -------------[STATE: START WRITE ADR BYTE CYCLE          ]-------------------
	;; -------------[DO: SET BDIR                               ]-------------------
	;; -------------[DO: SET BC1                                ]-------------------
	bsf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	;; SET BDIR (AY Chip)
	bsf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	;; SET BC1 (AY Chip)
	
	
	;; -------------[DO: WRITE 8BIT SERIAL DATA TO 74HC595      ]------------------------------
	;; -------------[USE: AY_SR_WRITE_BIT MACRO                 ]------------------------------
	;; Note: HAVE A LOOK @ AY_SR_WRITE_BIT MACRO for Details
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 0	;; BIT0 AY CHIP ADDR. BYTE => [B x x x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 1	;; BIT1 AY CHIP ADDR. BYTE => [x B x x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 2	;; BIT2 AY CHIP ADDR. BYTE => [x x B x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 3	;; BIT3 AY CHIP ADDR. BYTE => [x x x B x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 4	;; BIT4 AY CHIP ADDR. BYTE => [x x x x B x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 5	;; BIT5 AY CHIP ADDR. BYTE => [x x x x x B x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 6	;; BIT6 AY CHIP ADDR. BYTE => [x x x x x x B x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_ADDRESS, 7	;; BIT7 AY CHIP ADDR. BYTE => [x x x x x x x B]

	;; ------------[STATE: A SERIAL BYTE HAS BEEN TRANSMITTED  ]-------------------------------
	;; ------------[DO: STORE THIS BYTE IN THE 74595 LATCH     ]-------------------------------
    bsf 	AY_SR_LAT_RCLK, AY_SR_PIN_RCLK			; STORE AY CHIP ADR BYTE in 74HC595 LATCH
	bcf		AY_SR_LAT_SER_OUT, AY_SR_PIN_SER_OUT	; CLR SERIAL DATA OUT PIN (standby)
    bcf		AY_SR_LAT_RCLK, AY_SR_PIN_RCLK			; DO RELEASE LATCH (74HC595 HANDLING)
		

	;; --------------[STATE: AY CHIP ADR BYTE HAD BEEN STORED            ]---------------------
	;; --------------[DO: SET AY CHIP BUS INACTIVE (END ADR BYTE CYCLE)  ]---------------------
	;; AY BUS CONTROL: SET BDIR TO 0, SET BC1 TO 0
	bcf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	;; CLEAR BDIR (AY Chip Control)
	bcf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	;; CLEAR BC1  (AY Chip Control)

	
	;; ----[STATE: THE ADDRESS BYTE HAS NOW BEEN WRITTEN TO THE AY CHIP  ]---------------------
	;; ----[AY CHIP BUS CONTROL: DO SET BDIR TO 1, DO SET BC1 TO 0       ]---------------------
	bsf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	;; SET BDIR (AY Chip Control)
	bcf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	;; SET BC1 	(AY Chip Control)
	
	;; ---------[STATE: START WRITE DATA BYTE CYCLE                      ]---------------------
	;; ---------[DO: WRITE 8BIT SERIAL DATA TO 74HC595                   ]---------------------
	;; => SEE AY_SR_WRITE_BIT MACRO!
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 0	;; BIT0 AY CHIP	DATA BYTE => [B x x x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 1	;; BIT1 AY CHIP DATA BYTE => [x B x x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 2	;; BIT2 AY CHIP DATA BYTE => [x x B x x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 3	;; BIT3 AY CHIP DATA BYTE => [x x x B x x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 4	;; BIT4 AY CHIP DATA BYTE => [x x x x B x x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 5	;; BIT5 AY CHIP DATA BYTE => [x x x x x B x x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 6	;; BIT6 AY CHIP DATA BYTE => [x x x x x x B x]
	AY_SR_WRITE_BIT VAR_AY_CHIP_REG_DATA, 7	;; BIT7 AY CHIP DATA BYTE => [x x x x x x x B]

	;; ------------[STATE: A SERIAL BYTE HAS BEEN TRANSMITTED  ]-------------------------------
	;; ------------[DO: STORE THIS BYTE IN THE 74595 LATCH     ]-------------------------------
    bsf AY_SR_LAT_RCLK, AY_SR_PIN_RCLK			; STORE AY CHIP ADR BYTE in 74HC595 LATCH
	bcf	AY_SR_LAT_SER_OUT, AY_SR_PIN_SER_OUT	; CLR SERIAL DATA OUT PIN (standby)
    bcf	AY_SR_LAT_RCLK, AY_SR_PIN_RCLK			; DO RELEASE LATCH (74HC595 HANDLING)
	
	;; ------------[SET AY BUS INACTIVE (END DATA BYTE CYCLE)  ]-------------------------------
	;; ------------[DO: CLR BDIR                               ]-------------------------------
	;; ------------[DO: CLR BC1                                ]-------------------------------
	bcf	AY_SR_LAT_BDIR, AY_SR_PIN_BDIR	;; CLEAR BDIR 	(AY Chip Control)
	bcf	AY_SR_LAT_BC1, AY_SR_PIN_BC1	;; CLEAR BC1 	(AY Chip Control)
	
	;; NOW A BYTE HAS BEEN WRITTEN TO AN AY CHIP REGISTER
	;; ENABLE Interrupt Request (IRQ)
	IRQ_ENABLE		

	;; A BYTE HAS BEEN TRANSMITTED TO A AY-3-8912 REGISTER
	;; END OF THE SERIAL WRITING ROUTINE
    return
	
	;; **************************************************************************
	;; --------------------------------------------------------------------------
	;;           END OF: AY_SR_Write: WRITE A BYTE TO AN AY-3-8912 REGSITER
	;; --------------------------------------------------------------------------
	;; **************************************************************************


