/* 
 * =============================================================================
 * 															MIDIBOX PROJECT
 * =============================================================================
 */

// Headers noyau
#include "cmios.h"
#include "pic18f452.h"
#include "aerodrink.h"

// --------------------- Animation des bargraphs (screensaver) -------------------

#define SS_SIZE 4					// Nombre  de cycles du screensaver #1
#define SS_INV  -1 				// Inversion ou non du sens de dfilement

#define SCREEN_SAVER_REPEAT  4000 // Temps entre 2 conomiseurs en IDLE
#define SCREEN_SAVER_DELAY   2000 // Temps avant dclenchement de l'conomiseur (aprs toucher)
#define SCREEN_SAVER_LENGTH   700 // Dure de l'conomiseur
#define SCREEN_SAVER_CYCLE      3 // Temps entre 2 cycles d'affichages des bargraphs
unsigned char SS_CYCLE; 		       // Horloge de cycles (0,1,2,..,SS_SIZE-1,0,1,2,etc.)

unsigned long SS_DELAY; 	// Timer de dcompte avant dclenchement de l'conomiseur
unsigned long SS_LENGTH;	// Timer de dcompte avant l'arrt de l'conomiseur
unsigned long SS_STEP;  	// Compteur pour temporisation entre les cycles

#define JOYS_REFRESH  200 // Temps avant de rafraichir les BG avec les valeurs des joysticks centraux

// Dfinition des squences de codage des leds
const unsigned char SS_BG12[16]={204,192, 51,  0,102, 64,153,128, 51,  0,204,192,153,128,102, 64};
const unsigned char SS_BG34[16]={204,192, 51,  0,153,128,102, 64, 51,  0,204,192,102, 64,153,128};
const unsigned char SS_BG56[16]={204,  3, 51,  0,102,  2,153,  1, 51,  0,204,  3,153,  1,102,  2};
const unsigned char SS_BG78[16]={204,  3, 51,  0,102,  2,153,  1, 51,  0,204,  3,153,  1,102,  2};

//---------------------- D-Beam (capteurs infrarouge) --------------------------

unsigned char DBEAMS_ACTIVE; // Valable pour les 2 DBEAMs, vaut ON ou OFF
unsigned char DBEAM1_STATUS; // Vaut ON ou OFF
unsigned char DBEAM2_STATUS; // Vaut ON ou OFF

#define DBEAM_MAX       500  // Valeur AIN max (10 bits) pour atteindre 127 (500)
#define DBEAM_THRESHOLD 210  // Valeur AIN min (10 bits) en-dessous de laquelle (150)
                             // on considre le GP2D12 comme inactif

//---------------------- SoftPot (capteurs de pression) ------------------------

#define SOFTPOT1_THRESHOLD 10
#define SOFTPOT2_THRESHOLD 50 //50

unsigned char SOFTPOTS_ACTIVE; // Valable pour les 2 DBEAMs, vaut ON ou OFF
unsigned char SOFTPOT1_STATUS; // Vaut ON  ou OFF
unsigned char SOFTPOT2_STATUS; // Vaut ON  ou OFF

//---------------------- Pdale d'expression -----------------------------------

unsigned char EXPPEDAL_ACTIVE; // Prise en compte ou non de la pdale
                               // (interfrences si non connecte)

// --------------------- Dfinition des contrleurs MIDI -----------------------

#define SR_JOY 04 // Shit Register des joystiks

unsigned char BG1,EBG1,BG2,EBG2,BG3,EBG3,BG4,EBG4; // Bargraphs du joystick 1
unsigned char BG5,EBG5,BG6,EBG6,BG7,EBG7,BG8,EBG8; // Bargraphs du joystick 2

int TIMER_BG_LEFTJOYS;  // Timers (dlais) de raffichage sur les bargraphs des 
int TIMER_BG_RIGHTJOYS; // valeurs des joysticks analogiques

// --------------------- Prdfinition -----------------------------------------

void AIN_NotifyChange(unsigned char pin, unsigned int pin_value) __wparam;

// --------------------- Dfinition des caractres spciaux --------------------

const unsigned char CHARSET[8*8] = 
{
	 4,14,31, 0, 0, 0, 0, 0, // Flche vers le haut, pleine
	31,17,23,17,23,17,31,31, // Symbole "E" pour spcifier qu'il s'agit du preset en edition (MODE_BANK)
	 8,12,14,15,14,12, 8, 0, // Flche vers la droite, pleine
	24,20,18,17,17,18,20,24, // Flche vers la droite, creuse
	 2, 6,14,30,14, 6, 2, 0, // Flche vers la gauche, pleine
	 3, 5, 9,17,17, 9, 5, 3, // Flche vers la gauche, creuse
	31,29,29,27,27,23,23,31, // Symbole LIN
	31,29,29,29,29,27,23,31, // Symbole LOG
};

// --------------- Dclaration des fonctions lies aux bargraphs ----------------

unsigned char getBG1from7bits(unsigned char value)
{
		unsigned char returned;
		if      (value == 0)	 returned=0;
		else if (value <= 06)  returned=128;
		else if (value <= 13)  returned=192;
		else if (value <= 19)  returned=224;
		else if (value <= 26)  returned=240;
		else if (value <= 32)  returned=248;
		else if (value <= 38)  returned=252;
		else if (value <= 45)  returned=254;
		else                   returned=255;
		return(returned);
}
unsigned char getEBG1from7bits(unsigned char value)
{
		unsigned char returned;
		if      (value <= 51)  returned=0;
		else if (value <= 58)  returned=128; // On allume la led 9 du bargraph 1
		else                   returned=192; // On allume les leds 9 et 10 du bargraph 1
		return(returned);
}
unsigned char getXEBG1from7bits(unsigned char value)
{
		// Fonction dcline de getEBG1from7bits (car diffrent pour le joystick de PlayStation)
		unsigned char returned;
		if      (value <= 51)  returned=0;
		else if (value <= 58)  returned=1; // On allume la led 9 du bargraph 1
		else                   returned=3; // On allume les leds 9 et 10 du bargraph 1
		return(returned);
}
unsigned char getBG2from7bits(unsigned char value)
{
		unsigned char returned;
		if      (value <= 64)  returned=0;
		else if (value <= 70)  returned=128;
		else if (value <= 77)  returned=192;
		else if (value <= 83)  returned=224;
		else if (value <= 90)  returned=240;
		else if (value <= 96)  returned=248;
		else if (value <= 102) returned=252;
		else if (value <= 109) returned=254;
		else                   returned=255;
		return(returned);
}
unsigned char getEBG2from7bits(unsigned char value)
{
		unsigned char returned;
		if      (value <= 115)  returned=0;
		else if (value <= 121)  returned=128; // On allume la led 9 du bargraph 2
		else                    returned=192; // On allume les leds 9 et 10 du bargraph 2
		return(returned);
}
unsigned char getXEBG2from7bits(unsigned char value)
{
		// Fonction dcline de getEBG2from7bits (car diffrent pour le joystick de PlayStation)
		unsigned char returned;
		if      (value <= 115)  returned=0;
		else if (value <= 121)  returned=1; // On allume la led 9 du bargraph 2
		else                    returned=3; // On allume les leds 9 et 10 du bargraph 2
		return(returned);
}
void BG_DisplayLeftJoystick(unsigned char x,unsigned char y)
{
	// Les axes en 255 seront ignors (afichage d'un seul axe possible)
	
	// L'axe des X des joysticks gauches sont inverss (127  gauche) d'o le 127-x
	if (x!=255)
	{
		BG5  = getBG1from7bits(x);
		EBG5 = getXEBG1from7bits(x);
		BG6  = getBG2from7bits(x);
		EBG6 = getXEBG2from7bits(x);
	}

	if (y!=255)
	{
		BG7  = getBG1from7bits(y);
		EBG7 = getXEBG1from7bits(y);
		BG8  = getBG2from7bits(y);
		EBG8 = getXEBG2from7bits(y);
	}
}
void BG_DisplayRightJoystick(unsigned char x,unsigned char y)
{
	if (x!=255)
	{
		BG1  = getBG1from7bits(x);
		EBG1 = getEBG1from7bits(x);
		BG2  = getBG2from7bits(x);
		EBG2 = getEBG2from7bits(x);
	}
	if (y!=255)
	{
		BG3  = getBG1from7bits(y);
		EBG3 = getEBG1from7bits(y);
		BG4  = getBG2from7bits(y);
		EBG4 = getEBG2from7bits(y);
	}
}

// -------------------- Dclaration des fonctions du noyau ---------------------

void Init(void) __wparam
{
	// This function is called by MIOS after startup to initialize the application

	// Initialisations fonctionnelles
	MIOS_AIN_NumberSet(NB_AIN); // Pin max (connect) des entres AIN
	MIOS_AIN_Muxed();        		// Module AIN supplmentaire prsent dans le montage

	// La deadband AIN limite la rsolution des valeurs renvoyes  l'application (entre USER_AIN_NotifyPinChange). 
	// Pour les applications avec des venements MIDI 7bit (par exemple MIDIbox MF), une Deadband AIN de 7 doit tre 
	// spcifie.
	// (Note: 7 ne signifie pas 7bit, mais une deadband de +/- 7. Dans la mesure ou le MIOS travaille en interne avec une
	// rsolution de 10bit, cela signifie que les changements dans une bande de "3bit - 1" (2^3-1) sont ignors).
	// Toutes les autres applications Motorfader utilisent un protocole spcial qui supporte des rsolutions suprieures,
	// de fait une deadband AIN plus basse est prfrable . 
	// Une deadband de 0 entrane des valeurs instables (jitter) des valeurs de 1..3 seront plus adaptes.
	MIOS_AIN_DeadbandSet(7);

	// Initialisation des registres
	MIOS_SRIO_NumberSet(16);		// Shift Registers
	MIOS_SRIO_UpdateFrqSet(1);	// mS  

	// Activation de la fonction MIDI MERGE
	MIOS_MIDI_MergerSet(MIOS_MIDI_MERGER_ENABLED);

	// Espacement de dclenchement des appels automatiques  Timer()
	MIOS_TIMER_Init(3, 8192);
	
	TIMER_BG_LEFTJOYS  = 0;
	TIMER_BG_RIGHTJOYS = 0;

	// Dfinition des caractres spciaux utiliss	
	MIOS_CLCD_SpecialCharsInit(CHARSET);
	
	// SPECIAL_CALL est utilis lors de l'inversion, dans le cas des DBeams,
	// pour forcer l'envoi des valeurs de bord lors de l'appui sur Invert/Reset
	SPECIAL_CALL = 0;
	
	// Variable qui indique  tout moment si le bouton Frezze/Cut est enfonc ou pas
	BTN_FREEZE_PUSHED = 0;

	//-----------------------------------------------
	// Initialisation des tats / modes / prfrences
	//-----------------------------------------------

	// Beams :
	DBEAM1_STATUS=OFF;
	DBEAM2_STATUS=OFF;
	
	// Ribbons :
	SOFTPOT1_STATUS=OFF;
	SOFTPOT2_STATUS=OFF;

	DWSMALL=OFF; // Statut de l'appui sur les encoders
	DWBIG=OFF;	 // (on conserve l'tat pour le consulter lors du ENC_NotifyChange()
	SP_LASTNOTE[0]=-1;
	SP_LASTNOTE[1]=-1;
	
	// Dmarrage en mode VOICE :
	ADK_MODE = MODE_VOICE;
	GLOBAL_LAST_TOUCHED_PIN = 255; // Pas encore de controleur touch

	// Initialisation du compteur de PC
	GLOBAL_PROG_CHANGE	= 1;
	GLOBAL_PRESET_NO = 1;
	
	LOAD_PRESET(GLOBAL_PRESET_NO); // -1 : Default Preset / 1-128 : Standard preset

	//MIOS_MIDI_TxBufferPut(0xfa); 	// Midi Start
}
void Tick(void) __wparam 
{
	// This function is called by MIOS in the mainloop when nothing else is to do 

	//MIOS_MIDI_TxBufferPut(0xf8); 	// Midi Clock

}
void Timer(void) __wparam 
{
	// This function is periodically called by MIOS. The frequency has to be initialized with MIOS_Timer_Set
	if (TIMER_BG_LEFTJOYS>0)
	{
		TIMER_BG_LEFTJOYS = TIMER_BG_LEFTJOYS-1;
		if (TIMER_BG_LEFTJOYS==1)
		{
			// On rinitialise les 2 bargraphs aux positions des joysticks centraux
			unsigned char LX = TRANSFORM_VALUE(J1X_PIN, MIOS_AIN_Pin7bitGet(J1X_PIN));
			unsigned char LY = TRANSFORM_VALUE(J1Y_PIN, MIOS_AIN_Pin7bitGet(J1Y_PIN));
			BG_DisplayLeftJoystick(LX,LY);
			//BG_DisplayLeftJoystick(MIOS_AIN_Pin7bitGet(J1X_PIN),MIOS_AIN_Pin7bitGet(J1Y_PIN));
		}
	}
	if (TIMER_BG_RIGHTJOYS>0)
	{
		TIMER_BG_RIGHTJOYS = TIMER_BG_RIGHTJOYS-1;
		if (TIMER_BG_RIGHTJOYS==1)
		{
			// On rinitialise les 2 bargraphs aux positions des joysticks centraux
			unsigned char RX = TRANSFORM_VALUE(J2X_PIN, MIOS_AIN_Pin7bitGet(J2X_PIN));
			unsigned char RY = TRANSFORM_VALUE(J2Y_PIN, MIOS_AIN_Pin7bitGet(J2Y_PIN));
			BG_DisplayRightJoystick(RX,RY);
			//BG_DisplayRightJoystick(MIOS_AIN_Pin7bitGet(J2X_PIN),127-MIOS_AIN_Pin7bitGet(J2Y_PIN));
		}
	}
	
	//----------------------------------
	// Gestion de l'conomiseur d'cran
	//----------------------------------
	
	if (SS_DELAY>0)
	{
		SS_DELAY=SS_DELAY-1;
		
		// S'il est temps pour l'conomiseur de se dclencher
		if (SS_DELAY==0) 
		{
			// Initialisation du timer de dure
			SS_LENGTH=SCREEN_SAVER_LENGTH; 
		}
	}
	
	// Si le temps imparti  l'conomiseur n'est pas encore consomm
	if (SS_LENGTH>0)
	{
		// Si l'conomiseur vient juste d'tre dclench
		if (SS_LENGTH==SCREEN_SAVER_LENGTH)
		{
			SS_STEP=0;
			SS_CYCLE=0;
		}
		else
		{
			SS_STEP=SS_STEP+1;
			if (SS_STEP > SCREEN_SAVER_CYCLE) 
			{ 
				SS_CYCLE=SS_CYCLE+1;
				if (SS_CYCLE==SS_SIZE)
				{
					SS_CYCLE=0;
				}
				SS_STEP=0; 

			}
		}

		BG1  = SS_BG12[SS_CYCLE*SS_SIZE];
		EBG1 = SS_BG12[(SS_CYCLE*SS_SIZE)+1];
		BG2  = SS_BG12[(SS_CYCLE*SS_SIZE)+2];
		EBG2 = SS_BG12[(SS_CYCLE*SS_SIZE)+3];

		BG3  = SS_BG34[SS_CYCLE*SS_SIZE];
		EBG3 = SS_BG34[(SS_CYCLE*SS_SIZE)+1];
		BG4  = SS_BG34[(SS_CYCLE*SS_SIZE)+2];
		EBG4 = SS_BG34[(SS_CYCLE*SS_SIZE)+3];

		BG5  = SS_BG56[SS_CYCLE*SS_SIZE];
		EBG5 = SS_BG56[(SS_CYCLE*SS_SIZE)+1];
		BG6  = SS_BG56[(SS_CYCLE*SS_SIZE)+2];
		EBG6 = SS_BG56[(SS_CYCLE*SS_SIZE)+3];

		BG7  = SS_BG78[(SS_SIZE-1-SS_CYCLE)*SS_SIZE];
		EBG7 = SS_BG78[(SS_SIZE-1-SS_CYCLE)*SS_SIZE+1];
		BG8  = SS_BG78[(SS_SIZE-1-SS_CYCLE)*SS_SIZE+2];
		EBG8 = SS_BG78[(SS_SIZE-1-SS_CYCLE)*SS_SIZE+3];

		// Je dcrmente le compteur d'excution de l'conomiseur
		SS_LENGTH = SS_LENGTH-1;
		
		// S'il est arriv  terme, je raffiche les valeurs originales des joysticks
	  if (SS_LENGTH == 0)
	  {
			// On rinitialise les 2 bargraphs aux positions des joysticks centraux
			unsigned char LX = TRANSFORM_VALUE(J1X_PIN, MIOS_AIN_Pin7bitGet(J1X_PIN));
			unsigned char LY = TRANSFORM_VALUE(J1Y_PIN, MIOS_AIN_Pin7bitGet(J1Y_PIN));
			unsigned char RX = TRANSFORM_VALUE(J2X_PIN, MIOS_AIN_Pin7bitGet(J2X_PIN));
			unsigned char RY = TRANSFORM_VALUE(J2Y_PIN, MIOS_AIN_Pin7bitGet(J2Y_PIN));
			
			BG_DisplayLeftJoystick(LX,LY);
			BG_DisplayRightJoystick(RX,RY);
	  	
			SS_DELAY=SCREEN_SAVER_REPEAT;
	  }
	}
}
void DISPLAY_Init(void) __wparam 
{
	// This function is called by MIOS when the display content should be initialized. 
	// Thats the case during startup and after a temporary message has been printed on the screen

	// Affichage des positions des joysticks sur les bargraphs

	int valeur7bits;

	MIOS_MIDI_Init();
	BG_DisplayLeftJoystick(MIOS_AIN_Pin7bitGet(J1X_PIN),MIOS_AIN_Pin7bitGet(J1Y_PIN));
	BG_DisplayRightJoystick(MIOS_AIN_Pin7bitGet(J2X_PIN),127-MIOS_AIN_Pin7bitGet(J2Y_PIN));

	SS_DELAY=SCREEN_SAVER_REPEAT;
	MIOS_LCD_Clear();

	// DBeams :
	if (MIOS_DIN_PinGet(BTN_DBONOFF)==0) 
	{
		DBEAMS_ACTIVE=ON;
		MIOS_DOUT_PinSet(LED2_DB1,1);
		MIOS_DOUT_PinSet(LED2_DB2,1);
  }
  else
  {
		DBEAMS_ACTIVE=OFF;
		MIOS_DOUT_PinSet(LED2_DB1,0);
		MIOS_DOUT_PinSet(LED2_DB2,0);
	}

	// Ribbons :
	if (MIOS_DIN_PinGet(BTN_RIBBS)==1) 
	{
		SOFTPOTS_ACTIVE=ON;
		MIOS_DOUT_PinSet(LED_ADKL,1);
		MIOS_DOUT_PinSet(LED_ADKR,1);
	}
	else                               
	{
		SOFTPOTS_ACTIVE=OFF;
		MIOS_DOUT_PinSet(LED_ADKL,0);
		MIOS_DOUT_PinSet(LED_ADKR,0);
	}	
	
	// Par dfaut la pdale d'expression est inactive
	EXPPEDAL_ACTIVE=OFF;
	
	// Initialisation en mode VOICE :
	if (ADK_MODE == MODE_VOICE)
		MIOS_DOUT_PinSet(LED_VOICE,1);
	else
		MIOS_DOUT_PinSet(LED_VOICE,0);

	// Nouvel cran
	JUST_PLAY();
}
void DISPLAY_Tick(void) __wparam 
{
// This function is called in the mainloop when no temporary message is shown on screen
// Print the realtime messages here
}
void MPROC_NotifyReceivedEvnt(unsigned char evnt0, unsigned char evnt1, unsigned char evnt2) __wparam 
{
// This function is called by MIOS when a complete MIDI event has been received
}
void MPROC_NotifyFoundEvent(unsigned entry, unsigned char evnt0, unsigned char evnt1, unsigned char evnt2) __wparam 
{
	// This function is called by MIOS when a MIDI event has been received
	// which has been specified in the MIOS_MPROC_EVENT_TABLE
}
void MPROC_NotifyTimeout(void) __wparam 
{
	// This function is called by MIOS when a MIDI event has not been completly received within 2 seconds
}
void MPROC_NotifyReceivedByte(unsigned char byte) __wparam 
{
	// This function is called by MIOS when a MIDI byte has been received
}
void SR_Service_Prepare(void) __wparam 
{	
	// This function is called by MIOS before the shift register are loaded

	static unsigned char sr_ctr;

	sr_ctr = ++sr_ctr & 0x03;

	//   |---R3---R4
	//   R
	//   1
	//   |
	//   R
	//   2
	
	switch(sr_ctr)
	{
		case 0 :
		{
			MIOS_DOUT_SRSet(SR_JOY,255);
			MIOS_DOUT_PinSet0(J1R1);
			MIOS_DOUT_PinSet0(J2R1);

			MIOS_DOUT_SRSet(7,BG1);
			MIOS_DOUT_SRSet(6,EBG1+EBG5);
			MIOS_DOUT_SRSet(5,BG5);
			break;
		}
		case 1 :
		{
			MIOS_DOUT_SRSet(SR_JOY,255);
			MIOS_DOUT_PinSet0(J1R2);
			MIOS_DOUT_PinSet0(J2R2);

			MIOS_DOUT_SRSet(7,BG2);
			MIOS_DOUT_SRSet(6,EBG2+EBG6);
			MIOS_DOUT_SRSet(5,BG6);
			break;
		}
		case 2 :
		{
			MIOS_DOUT_SRSet(SR_JOY,255);
			MIOS_DOUT_PinSet0(J1R3);
			MIOS_DOUT_PinSet0(J2R3);

			MIOS_DOUT_SRSet(7,BG3);
			MIOS_DOUT_SRSet(6,EBG3+EBG7);
			MIOS_DOUT_SRSet(5,BG7);
			break;
		}
		case 3 :
		{
			MIOS_DOUT_SRSet(SR_JOY,255);
			MIOS_DOUT_PinSet0(J1R4);
			MIOS_DOUT_PinSet0(J2R4);

			MIOS_DOUT_SRSet(7,BG4);
			MIOS_DOUT_SRSet(6,EBG4+EBG8);
			MIOS_DOUT_SRSet(5,BG8);
			break;
		}
	}
}
unsigned int ADK_DIV(unsigned int a, unsigned int b)
{
	// Division entire (a div b)
  unsigned int reste = 0;
  unsigned char count = 16;
  char c;

  do
  {
    // reste: a <- 0;
    c = ((a >> (8*sizeof(a)-1)) & 1);
    a <<= 1;
    reste <<= 1;
    if (c) reste |= 1;
    if (reste >= b)
    {
      reste -= b;
      // a <- (result = 1)
      a |= 1;
    }
  }
  while (--count);
  return a;
}
void SR_Service_Finish(void) __wparam 
{
// This function is called by MIOS after the shift register have been loaded
}
void DIN_NotifyToggle(unsigned char pin, unsigned char pin_value) __wparam 
{
// This function is called by MIOS when an button has been toggled
// pin_value is 1 when button released, and 0 when button pressed

	// A GARDER (pratique)
	/*
	MIOS_LCD_CursorSet(POS_LINE_2);
	MIOS_LCD_PrintBCD3(pin);
	MIOS_LCD_CursorSet(POS_LINE_3);
	MIOS_LCD_PrintBCD3(pin_value);
	*/	

	switch(pin)
	{
		case BTN_EDIT :
		{
			if (pin_value==0)
			{
				unsigned char do_nothing=0;
				
				//if (ADK_MODE==MODE_VOICE) ADK_MODE=MODE_EDIT;
				if (	 GLOBAL_LAST_TOUCHED_PIN==SOFTPOT1_PIN
					  || GLOBAL_LAST_TOUCHED_PIN==SOFTPOT2_PIN)
				{
					if (ADK_MODE==MODE_EDIT) ADK_MODE=MODE_EDIT2;
					else ADK_MODE=MODE_EDIT;
				}
				else
				{
					if (ADK_MODE!=MODE_EDIT) ADK_MODE=MODE_EDIT;
					else do_nothing=1;
				}
				
				if (do_nothing==0)
				{
					if (ADK_MODE == MODE_EDIT)
					{
						// Nouvel cran
						MIOS_LCD_Clear();
						MIOS_LCD_CursorSet(POS_LINE_1);	MIOS_LCD_PrintCString("--------------------");
						MIOS_LCD_CursorSet(POS_LINE_2);	MIOS_LCD_PrintCString("TOUCH ANY CONTROLLER");
						MIOS_LCD_CursorSet(POS_LINE_3);	MIOS_LCD_PrintCString("--------------------");
	
						// On allume/teind les leds
						MIOS_DOUT_PinSet(LED_INTCRT,1);
						MIOS_DOUT_PinSet(LED_VOICE,0);
						MIOS_DOUT_PinSet(LED_BANK,0);
						MIOS_DOUT_PinSet(LED_INV,0); 		// On teind la led Invert/Reset

						// Je rinitialise la variable GLOBAL_LAST_TOUCHED_PIN 
						GLOBAL_LAST_TOUCHED_PIN = 255;
					}
					else if (ADK_MODE == MODE_EDIT2) // MODE EDITION AVANCEE
					{
						// Index pour les structures d'dition avance
						int adv_idx = 0; // correspond  SOFTPOT1_PIN
						if (GLOBAL_LAST_TOUCHED_PIN == SOFTPOT2_PIN) adv_idx = 1;
						
						// On conserve l'ID du controleur dit car les autres peuvent continuer de jouer
						GLOBAL_EDIT_ADV_PIN=GLOBAL_LAST_TOUCHED_PIN;
						GLOBAL_EDIT_ADV_IDX=GLOBAL_LAST_TOUCHED_IDX;
						
						// Nouvel cran d'dition avance
						MIOS_LCD_Clear();
						MIOS_LCD_CursorSet(POS_LINE_1);
						MIOS_LCD_PrintCString(CTL_NAME_PART1[GLOBAL_EDIT_ADV_IDX]);
						MIOS_LCD_PrintCString(CTL_NAME_PART2[GLOBAL_EDIT_ADV_IDX]);
						
						MIOS_LCD_CursorSet(POS_LINE_2);
						MIOS_LCD_PrintCString("Mode    :");
						if (SP_MODE[adv_idx] == SP_MODE_CC)
							MIOS_LCD_PrintCString("CC  ");
						else
							MIOS_LCD_PrintCString("NOTE");

						MIOS_LCD_CursorSet(POS_LINE_3);
						MIOS_LCD_PrintCString("Release :");
						if (SP_RELEASE[adv_idx] == SP_RELEASE_RTZ)
							MIOS_LCD_PrintCString("RTZ ");
						else
							MIOS_LCD_PrintCString("KEEP");
					}
				}
			}
			break;
		}
		case BTN_STORE :
		{
			if (pin_value==0)
			{
				// Organisation du bankstick : 
				// Le module 0 stocke le preset par dfaut ainsi que les paramtres globaux de l'Arkade
				// Les modules 1 et 2 stockent les presets
				// Chaque preset occupe 512 octets, chaque module fait 32 Ko (32768/512=64 presets)
				// Total=128 presets
			
				// Pour sauvegarder le preset courant comme tant celui par dfaut, 'presetno' doit valoir -1
				// Sinon les presets commencent  1 donc logiquement presetno ne vaut jamais zro
				
				// Slection du bankstick
				int presetno;									// Numro de preset qui doit tre utilis
				unsigned char presetname[16]; // Nom de preset qui doit tre utilis
				unsigned char bankstick;
				int mem_offset;
				
				unsigned char cpt=0;
				unsigned char ctl=0;
				unsigned char error = 0;
			
				// Si au moment o j'appuie sur Store le bouton Freeze/Cut est dj enfonc, le preset
				// en cours devient le preset par dfaut.
				if (BTN_FREEZE_PUSHED==1) presetno = -1;
					
				// En mode bank le preset que je vais craser est celui en cours de browse
				// (que a soit celui en cours ou pas)
				else if (ADK_MODE==MODE_BANK) // && GLOBAL_PRESET_NO!=GLOBAL_BROWSE_NO)
				{
					presetno = GLOBAL_BROWSE_NO;
					for (cpt=0; cpt<16; cpt++)
					{
						presetname[cpt]=GLOBAL_BROWSE_NAME[cpt];
						GLOBAL_PRESET_NAME[cpt] = GLOBAL_BROWSE_NAME[cpt];
					}
					// Le preset nouvellement cras devient celui charg
					GLOBAL_PRESET_NO = GLOBAL_BROWSE_NO;
				}
				// - Sinon je sauvegarde le preset actuel	
				else
				{
					presetno = GLOBAL_PRESET_NO;
					for (cpt=0; cpt<16; cpt++) presetname[cpt]=GLOBAL_PRESET_NAME[cpt];
				}
				
				// Slection du bankstick
				bankstick = GET_BANKSTICK(presetno); 
				
				// Slection de l'offset mmoire
				mem_offset = GET_MEM_OFFSET(presetno);
				
				if (presetno==0) return; // L'appel avec cette valeur ne doit pas se produire
			
				MEM_BUSY=1; // On flague pour que l'activit des controleurs ne perturbe pas l'I/O
			
				MIOS_LCD_Clear();
				MIOS_LCD_PrintCString("Writing memory ...");
			
				MIOS_BANKSTICK_CtrlSet(bankstick); // On slectionne le BankStick qui va bien
			
				// Ecriture des noms des controleurs
				// 17x16=272 octets (0 - 271)
				// Note : << 4 <==> multiplication par 16
				for (ctl=0; ctl<17; ctl++)
				{
					for (cpt=0; cpt<16; cpt++)
					{
						if (cpt<8)
							error |= MIOS_BANKSTICK_Write(mem_offset+(ctl << 4)+cpt,CTL_NAME_PART1[ctl][cpt]);
						else
							error |= MIOS_BANKSTICK_Write(mem_offset+(ctl << 4)+cpt,CTL_NAME_PART2[ctl][cpt-8]);
					}
				}
				
				// Ecriture des n de CC
				// 17x1=17 octets (272 - 288)
				for (ctl=0; ctl<17; ctl++)
				{
					error |= MIOS_BANKSTICK_Write(mem_offset+272+ctl,CTL_CC[ctl]);
				}
				
				// Ecriture des canaux MIDI
				// 17x1=17 octets (289 - 305)
				for (ctl=0; ctl<17; ctl++)
				{
					error |= MIOS_BANKSTICK_Write(mem_offset+289+ctl,CTL_MIDICH[ctl]);
				}
				
				// Ecriture des inversions
				// 17x1=17 octets (306 - 322)
				for (ctl=0; ctl<17; ctl++)
				{
					error |= MIOS_BANKSTICK_Write(mem_offset+306+ctl,CTL_INV[ctl]);
				}
			
				// Ecriture des Mins
				// 17x1=17 octets (323 - 339)
				for (ctl=0; ctl<17; ctl++)
				{
					error |= MIOS_BANKSTICK_Write(mem_offset+323+ctl,CTL_MIN[ctl]);
				}
			
				// Ecriture des Maxs
				// 17x1=17 octets (340 - 356)
				for (ctl=0; ctl<17; ctl++)
				{
					error |= MIOS_BANKSTICK_Write(mem_offset+340+ctl,CTL_MAX[ctl]);
				}
			
				// Ecriture des LIN/LOG (DBeams)
				// 2 octets (357 - 358)
				error |= MIOS_BANKSTICK_Write(mem_offset+357,DB_LINLOG[0]);
				error |= MIOS_BANKSTICK_Write(mem_offset+358,DB_LINLOG[1]);
			
				// Ecriture des modes (CC/NOTE) et release (RTZ/KEEP) pour les SoftPots
				// 4 octets (359 - 362)
				error |= MIOS_BANKSTICK_Write(mem_offset+359,SP_MODE[0]);
				error |= MIOS_BANKSTICK_Write(mem_offset+360,SP_RELEASE[0]);
				error |= MIOS_BANKSTICK_Write(mem_offset+361,SP_MODE[1]);
				error |= MIOS_BANKSTICK_Write(mem_offset+362,SP_RELEASE[1]);
				
				// Ecriture du nom du preset (sauf le default qui ne change pas)
				if (presetno!=-1)
				{
					for (cpt=0; cpt<16; cpt++)
						error |= MIOS_BANKSTICK_Write(mem_offset+363+cpt,presetname[cpt]);
				}
								
				MIOS_LCD_Clear();
				if (error) 
				{
					MIOS_LCD_CursorSet(POS_LINE_1);
					MIOS_LCD_PrintCString("Erreur BankStick !");
					MIOS_LCD_CursorSet(POS_LINE_2);
					MIOS_LCD_PrintCString("Error code : ");
					MIOS_LCD_PrintBCD3(error);
				}
				else
				{
					JUST_PLAY();
					MIOS_LCD_CursorSet(POS_LINE_3);
					MIOS_LCD_PrintCString("   ");
					MIOS_LCD_PrintChar(4);
					if (presetno==-1)
					{
						MIOS_LCD_PrintCString("Default updated");
						MIOS_LCD_PrintChar(2);
					}
					else
					{
						MIOS_LCD_PrintCString("Preset saved");
						MIOS_LCD_PrintChar(2);
						MIOS_LCD_PrintCString("   ");
					}
				}
			
				MEM_BUSY=0;
				GLOBAL_LAST_TOUCHED_PIN=255;

				// On passe en mode voice				
				ADK_MODE=MODE_VOICE;
				
				// On allume/teind les leds
				MIOS_DOUT_PinSet(LED_VOICE,1); 	// On allume la led Voice
				MIOS_DOUT_PinSet(LED_INTCRT,0);	// On teind la led INT/CRT
				MIOS_DOUT_PinSet(LED_BANK,0);
				MIOS_DOUT_PinSet(LED_INV,0); 		// On teind la led Invert/Reset
				}
			break;
		}
		case BTN_PEDDOWN :
		{
			if (pin_value==0)
				MIOS_DOUT_PinSet(LED_ARROWL,1);
			else
				MIOS_DOUT_PinSet(LED_ARROWL,0);
			break;
		}
		case BTN_PEDUP :
		{
			if (pin_value==0)
				MIOS_DOUT_PinSet(LED_ARROWR,1);
			else
				MIOS_DOUT_PinSet(LED_ARROWR,0);
			break;
		}
		case BTN_ARROWL :
		{
			if (pin_value==0)
			{
				MIOS_DOUT_PinSet(LED_ARROWL,1);

				if (ADK_MODE == MODE_EDIT && GLOBAL_LAST_TOUCHED_PIN != 255)
				{
					// J'affiche la flche sous le nom (edition du nom)
					if (EDIT_POS>0) 
					{
						EDIT_POS--;
						MIOS_LCD_CursorSet(POS_LINE_2);
						MIOS_LCD_PrintCString("                ");
						MIOS_LCD_CursorSet(POS_LINE_2 + EDIT_POS);	
						MIOS_LCD_PrintChar(0);
					}
				}
				else if (ADK_MODE == MODE_BANK)
				{
					if (EDIT_POS>0) 
					{
						EDIT_POS--;
						MIOS_LCD_CursorSet(POS_LINE_2+4);
						MIOS_LCD_PrintCString("                ");
						MIOS_LCD_CursorSet(POS_LINE_2 + 4 + EDIT_POS);	
						MIOS_LCD_PrintChar(0);
					}
				}
			}
			else
				MIOS_DOUT_PinSet(LED_ARROWL,0);
			break;
		}
		case BTN_ARROWR :
		{
			if (pin_value==0)
			{
				MIOS_DOUT_PinSet(LED_ARROWR,1);

				if (ADK_MODE == MODE_EDIT && GLOBAL_LAST_TOUCHED_PIN != 255)
				{
					// J'affiche la flche sous le nom (edition du nom)
					if (EDIT_POS<15) 
					{
						EDIT_POS++;
						MIOS_LCD_CursorSet(POS_LINE_2);
						MIOS_LCD_PrintCString("                ");
						MIOS_LCD_CursorSet(POS_LINE_2 + EDIT_POS);	
						MIOS_LCD_PrintChar(0);
					}
				}
				else if (ADK_MODE == MODE_BANK)
				{
					if (EDIT_POS<15) 
					{
						EDIT_POS++;
						MIOS_LCD_CursorSet(POS_LINE_2+4);
						MIOS_LCD_PrintCString("                ");
						MIOS_LCD_CursorSet(POS_LINE_2 + 4 + EDIT_POS);	
						MIOS_LCD_PrintChar(0);
					}
				}
			}
			else
				MIOS_DOUT_PinSet(LED_ARROWR,0);
			break;
		}
		case BTN_BANK :
		{
			if (pin_value==0)
			{
				if (ADK_MODE!=MODE_BANK) 
				{
					int i;
					
					ADK_MODE = MODE_BANK;
				
					MIOS_DOUT_PinSet(LED_BANK,1);
					MIOS_DOUT_PinSet(LED_INTCRT,0);
					MIOS_DOUT_PinSet(LED_VOICE,0);
					MIOS_DOUT_PinSet(LED_INV,0); 		// On teind la led Invert/Reset

					GLOBAL_BROWSE_NO = GLOBAL_PRESET_NO;
					
					SCREEN_BANK();
				}
			}
			/*
			else
				MIOS_DOUT_PinSet(LED_BANK,0);*/
			break;
		}
		case BTN_INTCRT :
		{
			if (pin_value==0)
				MIOS_DOUT_PinSet(LED_INTCRT,1);
			else
				MIOS_DOUT_PinSet(LED_INTCRT,0);
			break;
		}
		case BTN_VOICE :
		{
			if (pin_value == 0)
			{
				if (ADK_MODE != MODE_VOICE)
				{
					// Passage en mode voice
					ADK_MODE = MODE_VOICE;
					
					GLOBAL_LAST_TOUCHED_PIN = 255;

					// Nouvel cran
					JUST_PLAY();
					
					// On allume/teind les leds
					MIOS_DOUT_PinSet(LED_VOICE,1); 	// On allume la led Voice
					MIOS_DOUT_PinSet(LED_INTCRT,0);	// On teind la led INT/CRT
					MIOS_DOUT_PinSet(LED_BANK,0);
					MIOS_DOUT_PinSet(LED_INV,0); 		// On teind la led Invert/Reset
				}
			}
			break;
		}
		case BTN_PERF :
		{
			if (pin_value==0)
			{
				if (EXPPEDAL_ACTIVE==ON)
				{
					EXPPEDAL_ACTIVE=OFF;
					MIOS_DOUT_PinSet(LED_PERF,0);//LED_PERF
				}
				else
				{
					EXPPEDAL_ACTIVE=ON;
					MIOS_DOUT_PinSet(LED_PERF,1);//LED_PERF
				}
			}
			break;
		}
		case BTN_INV :
		{
			int ain_pin;
			
			if (pin_value==0) // Sur appui
			{
				// On accde  l'inversion qu'on soit en mode VOICE ou EDIT 
				if (CTL_INV[GLOBAL_LAST_TOUCHED_IDX]==ON)
				{
					CTL_INV[GLOBAL_LAST_TOUCHED_IDX]=OFF;
					MIOS_DOUT_PinSet(LED_INV,0);
				}
				else
				{
					CTL_INV[GLOBAL_LAST_TOUCHED_IDX]=ON;
					MIOS_DOUT_PinSet(LED_INV,1);
				}
				
				// Dans le cas des DBeams, on notifie l'appel "spcial"
				ain_pin = GET_PIN_FROM_INDEX(GLOBAL_LAST_TOUCHED_IDX);
				
				// On simule l'envoi de la valeur :
				// SPECIAL_CALL = 1 signifie l'appel depuis BTN_INV
				SPECIAL_CALL = 1; 

				//if (CTL_INV[GLOBAL_LAST_TOUCHED_IDX]==ON)
					//AIN_NotifyChange(ain_pin,1023 - MIOS_AIN_PinGet(ain_pin));
				//else
					AIN_NotifyChange(ain_pin,MIOS_AIN_PinGet(ain_pin));

				SPECIAL_CALL=0; // Je rinitialise mon flag
			}
			break;
		}
		case BTN_TAP :
		{
			if (pin_value==0) 
			{
				MIOS_DOUT_PinSet(LED_TAP,1);
				
				if (ADK_MODE==MODE_EDIT)
				{
					if (CTL_LAST_VALUE[GLOBAL_LAST_TOUCHED_IDX] > CTL_MIN[GLOBAL_LAST_TOUCHED_IDX])
					{
					// On fixe la valeur du Max sur la valeur courante
					CTL_MAX[GLOBAL_LAST_TOUCHED_IDX] = CTL_LAST_VALUE[GLOBAL_LAST_TOUCHED_IDX];
				
					MIOS_LCD_CursorSet(POS_LINE_4+13);
					MIOS_LCD_PrintBCD3(CTL_MAX[GLOBAL_LAST_TOUCHED_IDX]);
					}
				}
			}
			else              
				MIOS_DOUT_PinSet(LED_TAP,0);
				
			break;
		}
		case BTN_CLONE :
		{
			if (pin_value==0) 
			{	
				MIOS_DOUT_PinSet(LED_CLONE,1);
				
				// En mode EDIT sur un DBeam, le bouton clone sert  fixer
				// la courbe de rponse (LIN / LOG)
				if (ADK_MODE==MODE_EDIT && (GLOBAL_LAST_TOUCHED_PIN==DBEAM1_PIN || GLOBAL_LAST_TOUCHED_PIN==DBEAM2_PIN))
				{
					// Index spcifique pour les tableaux de courbes de rponse
					int adv_idx = 0; // correspond  DBEAM1_PIN
					if (GLOBAL_LAST_TOUCHED_PIN==DBEAM2_PIN) adv_idx = 1;
					
					if (DB_LINLOG[adv_idx]==LIN) // Je passe en LOG
					{
						DB_LINLOG[adv_idx]=LOG;
						MIOS_LCD_CursorSet(POS_LINE_2+16);
						MIOS_LCD_PrintChar(7);
					}
					else // Je passe en LIN
					{
						DB_LINLOG[adv_idx]=LIN;
						MIOS_LCD_CursorSet(POS_LINE_2+16);
						MIOS_LCD_PrintChar(6);
					}
				}
			}
			else              MIOS_DOUT_PinSet(LED_CLONE,0);
			break;
		}
		case BTN_BEAM :
		{
			if (pin_value==0) 
			{
				MIOS_DOUT_PinSet(LED_BEAM,1);
				
				if (ADK_MODE==MODE_EDIT)
				{
					if (CTL_LAST_VALUE[GLOBAL_LAST_TOUCHED_IDX] < CTL_MAX[GLOBAL_LAST_TOUCHED_IDX])
					{
					// On fixe la valeur du Min sur la valeur courante
					CTL_MIN[GLOBAL_LAST_TOUCHED_IDX] = CTL_LAST_VALUE[GLOBAL_LAST_TOUCHED_IDX];
				
					MIOS_LCD_CursorSet(POS_LINE_4+4);
					MIOS_LCD_PrintBCD3(CTL_MIN[GLOBAL_LAST_TOUCHED_IDX]);
					}
				}
			}
			else              
				MIOS_DOUT_PinSet(LED_BEAM,0);
			break;
		}
		case BTN_RIBBS :
		{
			if (pin_value==1) 
			{
				SOFTPOTS_ACTIVE=ON;
				MIOS_DOUT_PinSet(LED_ADKL,1);
				MIOS_DOUT_PinSet(LED_ADKR,1);
			}
			else
			{
				SOFTPOTS_ACTIVE=OFF;
				MIOS_DOUT_PinSet(LED_ADKL,0);
				MIOS_DOUT_PinSet(LED_ADKR,0);
			}
			break;
		}
		case BTN_DBONOFF :
		{
			if (pin_value==0)
			{
				DBEAMS_ACTIVE=ON;
				MIOS_DOUT_PinSet(LED2_DB1,1);
				MIOS_DOUT_PinSet(LED2_DB2,1);
			}
			else
			{
				DBEAMS_ACTIVE=OFF;
				DBEAM1_STATUS=OFF;
				DBEAM2_STATUS=OFF;
				MIOS_DOUT_PinSet(LED1_DB1,0);
				MIOS_DOUT_PinSet(LED1_DB2,0);
				MIOS_DOUT_PinSet(LED2_DB1,0);
				MIOS_DOUT_PinSet(LED2_DB2,0);
			}
			break;
		}
		case BTN_FREEZE :
		{
			int ain_pin = GET_PIN_FROM_INDEX(GLOBAL_LAST_TOUCHED_IDX);;

			// On notifie l'appel "spcial" de faon  forcer l'envoi de la borne
			// SPECIAL_CALL = 2 signifie l'appel depuis BTN_FREEZE
		  SPECIAL_CALL = 2;

			// On simule l'envoi de la valeur :
			if (pin_value==0) // Sur appui
			{
				BTN_FREEZE_PUSHED=1;

				if (CTL_INV[GLOBAL_LAST_TOUCHED_IDX]==ON)
				{
					if (ain_pin==DBEAM1_PIN || ain_pin==DBEAM2_PIN)
						AIN_NotifyChange(ain_pin,DBEAM_MAX);
					else
						AIN_NotifyChange(ain_pin,1023);
				}
				else
				{
					if (ain_pin==DBEAM1_PIN || ain_pin==DBEAM2_PIN)
						AIN_NotifyChange(ain_pin,DBEAM_THRESHOLD);
					else
						AIN_NotifyChange(ain_pin,0);
				}
			}
			else // Sur relchement
			{
				BTN_FREEZE_PUSHED=0;
				AIN_NotifyChange(ain_pin, MIOS_AIN_PinGet(ain_pin));

				// On rinitialise le compteur qui refreshe la position des joysticks
	    	//TIMER_BG_LEFTJOYS=JOYS_REFRESH;
	    	//TIMER_BG_RIGHTJOYS=JOYS_REFRESH;
			}
			SPECIAL_CALL = 0;
			
			break;
		}
		case BTN_DWSMALL :
		{
			if (pin_value==0)
			{
				DWSMALL=ON;

				// En MODE_BANK, l'appui sur la roue 'Program Change' charge le preset parcouru				
				if (ADK_MODE == MODE_BANK)
				{
					LOAD_PRESET(GLOBAL_BROWSE_NO);
					// Le preset en cours d'dition devient celui qui tait parcouru
					GLOBAL_PRESET_NO = GLOBAL_BROWSE_NO;
					SCREEN_BANK();
				}
			}
			else
			{
				DWSMALL=OFF;
			}
			break;
		}
		case BTN_DWBIG :
		{
			if (pin_value==0)
				DWBIG=ON;
			else
				DWBIG=OFF;
			break;
		}
		case BTN_INC :
		{
			if (pin_value==0 && ADK_MODE==MODE_VOICE) ADK_ProgCh_INC();
			break;
		}
		case BTN_DEC :
		{
			if (pin_value==0 && ADK_MODE==MODE_VOICE) ADK_ProgCh_DEC();
			break;
		}
	}
}
void ENC_NotifyChange(unsigned char encoder, char incrementer) __wparam 
{
// This function is called by MIOS when an encoder has been moved
// incrementer is positive when encoder has been turned clockwise, else it is negative

// encoder : le gros = 1 / le petit = 0

	switch (ADK_MODE)
	{
		case MODE_VOICE:
		{
			if (incrementer<0)
			  ADK_ProgCh_INC();
			else
			  ADK_ProgCh_DEC();
			break;
		}
		case MODE_EDIT:
		{
			// La petite data wheel sert  changer le n de controleur MIDI
			// La grosse data wheel sert  changer le nom du controleur
			
			if (GLOBAL_LAST_TOUCHED_PIN!=255) // Si un controleur a bien t touch
			{
				if (encoder == DATAWHEEL_SMALL)
				{			
					if (DWSMALL == ON) // Si appui sur la molette en mme temps que l'on a tourn
					{
						// Je change le canal MIDI associ  ce capteur
						unsigned char ch = CTL_MIDICH[GLOBAL_LAST_TOUCHED_IDX];
						if (incrementer<0 && ch<16) ch++;
						else if (incrementer>0 && ch>1) ch--;
						CTL_MIDICH[GLOBAL_LAST_TOUCHED_IDX] = ch;
			
						// J'affiche le nouveau canal sur le LCD
						MIOS_LCD_CursorSet(POS_LINE_3+17);
						MIOS_LCD_PrintBCD3(ch);
						
					}
					else
					{
						// Je change le numro de controleur MIDI associ  ce capteur
						unsigned char cc = CTL_CC[GLOBAL_LAST_TOUCHED_IDX];
						if (incrementer<0 && cc<127) cc++;
						else if (incrementer>0 && cc>0) cc--;
						CTL_CC[GLOBAL_LAST_TOUCHED_IDX] = cc;
			
						// J'affiche le nouveau numro de controleur MIDI sur le LCD
						MIOS_LCD_CursorSet(POS_LINE_1+17);
						MIOS_LCD_PrintBCD3(cc);
					}
				}
				else
				{
					unsigned char lettre = ' ';
					if (EDIT_POS<8)
						lettre = CTL_NAME_PART1[GLOBAL_LAST_TOUCHED_IDX][EDIT_POS];
					else
						lettre = CTL_NAME_PART2[GLOBAL_LAST_TOUCHED_IDX][EDIT_POS-8];
						
					if (incrementer<0 && lettre<122)
					{
						lettre++;
					}
					else if (incrementer>0 && lettre>32) 
					{
						lettre--;
					}

					// On met  jour le bon tableau :
					if (EDIT_POS<8)
						CTL_NAME_PART1[GLOBAL_LAST_TOUCHED_IDX][EDIT_POS]=lettre;
					else
						CTL_NAME_PART2[GLOBAL_LAST_TOUCHED_IDX][EDIT_POS-8]=lettre;

					// J'affiche la nouvelle lettre 
					MIOS_LCD_CursorSet(POS_LINE_1+EDIT_POS);
					MIOS_LCD_PrintChar(lettre);
				}
			}
			break;
		}
		case MODE_EDIT2:
		{
			// Index pour les structures d'dition avance
			int adv_idx = 0;
			if (GLOBAL_LAST_TOUCHED_PIN==SOFTPOT2_PIN) adv_idx=1;

			if (encoder == DATAWHEEL_BIG)
			{			
				if (incrementer<0 && SP_MODE[adv_idx]==SP_MODE_CC)
					SP_MODE[adv_idx]=SP_MODE_NOTE;

				if (incrementer>0 && SP_MODE[adv_idx]==SP_MODE_NOTE)
					SP_MODE[adv_idx]=SP_MODE_CC;

				MIOS_LCD_CursorSet(POS_LINE_2);
				MIOS_LCD_PrintCString("Mode    :");
				if (SP_MODE[adv_idx] == SP_MODE_CC)
					MIOS_LCD_PrintCString("CC  ");
				else
					MIOS_LCD_PrintCString("NOTE");
			}
			else if (encoder == DATAWHEEL_SMALL)
			{
				if (incrementer<0 && SP_RELEASE[adv_idx]==SP_RELEASE_RTZ)
					SP_RELEASE[adv_idx]=SP_RELEASE_KEEP;

				if (incrementer>0 && SP_RELEASE[adv_idx]==SP_RELEASE_KEEP)
					SP_RELEASE[adv_idx]=SP_RELEASE_RTZ;
							
				MIOS_LCD_CursorSet(POS_LINE_3);
				MIOS_LCD_PrintCString("Release :");
				if (SP_RELEASE[adv_idx] == SP_RELEASE_RTZ)
					MIOS_LCD_PrintCString("RTZ ");
				else
					MIOS_LCD_PrintCString("KEEP");
			}
			break;
		}
		case MODE_BANK:
		{
			if (encoder == DATAWHEEL_SMALL)
			{
				if (incrementer<0) // +
				{
					if (GLOBAL_BROWSE_NO<128)
					{
						GLOBAL_BROWSE_NO++;
						SCREEN_BANK();
					}
				}
				else // -
				{
					if (GLOBAL_BROWSE_NO>1)
					{
						GLOBAL_BROWSE_NO--;
						SCREEN_BANK();
					}
				}
			}
			else
			{
				// La grande roue permet de changer les lettres du nom de preset
				unsigned char lettre = GLOBAL_BROWSE_NAME[EDIT_POS];

				if (incrementer<0 && lettre<122)	lettre++;
				else if (incrementer>0 && lettre>32) lettre--;
				
				GLOBAL_BROWSE_NAME[EDIT_POS]=lettre;
				
				// J'affiche la nouvelle lettre 
				MIOS_LCD_CursorSet(POS_LINE_1 + 4 + EDIT_POS);
				MIOS_LCD_PrintChar(lettre);				
			}
			break;
		}
	}
}
void AIN_NotifyChange(unsigned char pin, unsigned int pin_value) __wparam
{
	// This function is called by MIOS when a pot has been moved

	int	DBEAM_VALUE, SOFTPOT_VALUE;
	unsigned char DBEAM_SHUTUP = OFF;
	unsigned char SOFTPOT_SHUTUP = OFF;
	unsigned char idx = GET_INDEX_FROM_PIN(pin);

	unsigned char filtertest;

	unsigned int  valeur10bits = pin_value; //MIOS_AIN_PinGet(pin);
	unsigned char valeur7bits  = pin_value >> 3;//MIOS_AIN_Pin7bitGet(pin);  // 10 bits -> 7 bits

	unsigned int range;

	// Affichage des vnements
	/*
	if (pin!=DBEAM1_PIN && pin!=DBEAM2_PIN && pin!=SOFTPOT1_PIN && pin!=SOFTPOT2_PIN && pin!=EXPPED_PIN)
	{
		MIOS_LCD_CursorSet(POS_LINE_4);
		MIOS_LCD_PrintBCD3(pin);
		MIOS_LCD_PrintCString(":");
		MIOS_LCD_PrintBCD3(valeur7bits);
	}
	*/

  //---------------------------------------------------------------------------------
	// Etape 1 : Trouver la valeur 7 bits, ventuellement corrige 
  //           (Lin / Log / correction de polarit cause soudure inverse)
  //---------------------------------------------------------------------------------

	switch(pin)
	{
		case SOFTPOT1_PIN :
		{
			if (SOFTPOTS_ACTIVE==ON)
			{
				SOFTPOT_VALUE=valeur10bits-SOFTPOT1_THRESHOLD;
		  	if (SOFTPOT_VALUE<0) SOFTPOT_VALUE=0;	  	
		  	if (SOFTPOT_VALUE!=0)
		  	{
					SOFTPOT1_STATUS=ON;
					SOFTPOT_VALUE=ADK_DIV(14000,SOFTPOT_VALUE+80)-12;
					if (SOFTPOT_VALUE<0) SOFTPOT_VALUE=0;
					else if (SOFTPOT_VALUE>127) SOFTPOT_VALUE=127;
					
					//ALL_NOTES_OFF();
					//NOTE_ON(SOFTPOT_VALUE);
					
			    valeur7bits = 127-SOFTPOT_VALUE; // Correction soudure
				}
				else
				{
					if (SOFTPOT1_STATUS==ON)
					{ 
						// Il faut l'teindre
						SOFTPOT1_STATUS=OFF;
						
						SOFTPOT_SHUTUP=ON;
						//ALL_NOTES_OFF();
					}
				}
			}
			break;
		}
		case SOFTPOT2_PIN :
		{
			if (SOFTPOTS_ACTIVE==ON)
			{
				SOFTPOT_VALUE=valeur10bits-SOFTPOT2_THRESHOLD;
		  	if (SOFTPOT_VALUE<0) SOFTPOT_VALUE=0;	  	
		  	if (SOFTPOT_VALUE!=0)
		  	{
					SOFTPOT2_STATUS=ON;
					SOFTPOT_VALUE=ADK_DIV(21590,SOFTPOT_VALUE+123)-20; // 4318*5=21590
					if (SOFTPOT_VALUE<0) SOFTPOT_VALUE=0;
					else if (SOFTPOT_VALUE>127) SOFTPOT_VALUE=127;
					
					//ALL_NOTES_OFF();
					//NOTE_ON(SOFTPOT_VALUE);
			    
			    valeur7bits = 127-SOFTPOT_VALUE; // Correction soudure
				}
				else
				{
					if (SOFTPOT2_STATUS==ON)
					{ 
						// Il faut l'teindre
						SOFTPOT2_STATUS=OFF;
						
						SOFTPOT_SHUTUP = ON;
						//ALL_NOTES_OFF();
					}
				}
			}
			break;
		}
		case DBEAM1_PIN : case DBEAM2_PIN :
		{
			// Fonctionnement des leds :
			// Si les DBEAMs sont ON mais inactifs, orange=on  jaune=off
			// Si les DBEAMs sont ON et valeur<=10, orange=on  jaune=on
			// Si les DBEAMs sont ON et valeur>10,  orange=off jaune=on
			if (DBEAMS_ACTIVE==ON)
			{
				if (valeur10bits >= DBEAM_THRESHOLD && valeur10bits <= DBEAM_MAX)
				{	
					// S'il ne l'tait pas, je considre maintenant le DBEAM comme actif
					if (pin==DBEAM1_PIN && DBEAM1_STATUS==OFF) 
					{
						DBEAM1_STATUS=ON;
						MIOS_DOUT_PinSet(LED1_DB1,1);
					}
					if (pin==DBEAM2_PIN && DBEAM2_STATUS==OFF) 
					{
						DBEAM2_STATUS=ON;
						MIOS_DOUT_PinSet(LED1_DB2,1);
					}
					
					// Calculs suivant les modes (linaire ou logarithmique)
					if ((pin==DBEAM1_PIN && DB_LINLOG[0]==LIN)	|| (pin==DBEAM2_PIN && DB_LINLOG[1]==LIN))
					{
						// MODE LINEAIRE
						//MIOS_LCD_CursorSet(POS_LINE_1);
						//MIOS_LCD_PrintChar(2);
						//MIOS_LCD_PrintBCD5(valeur10bits);
						DBEAM_VALUE = 127-(ADK_DIV(48000,valeur10bits-10)-102); // 30000 +10 -61
					}
					else
					{
						// MODE LOGARITHMIQUE (natif du GP2D120)
						DBEAM_VALUE = valeur10bits-DBEAM_THRESHOLD;
						DBEAM_VALUE = (DBEAM_VALUE << 7)-DBEAM_VALUE; // x 127
						DBEAM_VALUE = ADK_DIV(DBEAM_VALUE,DBEAM_MAX-DBEAM_THRESHOLD-30);
					}
		
					// Rsolution des effets de bord
					if (DBEAM_VALUE<0) DBEAM_VALUE=0;
					if (DBEAM_VALUE>127) DBEAM_VALUE=127;

					// Si je suis en-dessous de 10 j'allume la led orange
					// sinon je l'teinds
					if (DBEAM_VALUE>10)
					{
						if (pin==DBEAM1_PIN) MIOS_DOUT_PinSet(LED2_DB1,0);
						if (pin==DBEAM2_PIN) MIOS_DOUT_PinSet(LED2_DB2,0);
					}
					else
					{
						if (pin==DBEAM1_PIN) MIOS_DOUT_PinSet(LED2_DB1,1);
						if (pin==DBEAM2_PIN) MIOS_DOUT_PinSet(LED2_DB2,1);
					}
					valeur7bits = DBEAM_VALUE;
				}
				else if ((pin==DBEAM1_PIN && DBEAM1_STATUS==ON) || (pin==DBEAM2_PIN && DBEAM2_STATUS==ON))
				{
					// Je considre le DBEAM comme inactif
					if (pin==DBEAM1_PIN)
					{
						DBEAM1_STATUS=OFF;
	
						MIOS_DOUT_PinSet(LED1_DB1,0); // Extinction de la led jaune
						MIOS_DOUT_PinSet(LED2_DB1,1); // Rallumage  de la led orange
		
						// Valeur => Bargraphs	
						BG_DisplayLeftJoystick(0,0);
					}
					else
					{
						DBEAM2_STATUS=OFF;
	
						MIOS_DOUT_PinSet(LED1_DB2,0); // Extinction de la led jaune
						MIOS_DOUT_PinSet(LED2_DB2,1); // Rallumage  de la led orange
	
						// Valeur => Bargraphs	
						BG_DisplayRightJoystick(0,0);
					}
			    // La valeur de bord doit tre envoye
			    DBEAM_SHUTUP=ON;
					// Valeur de bord
					valeur7bits=0;
				}
			}
			break;
		}
	}
	
	// Pas de shutup si je suis en mode release=keep :
	if (pin==SOFTPOT1_PIN && SP_RELEASE[0]==SP_RELEASE_KEEP) SOFTPOT_SHUTUP=OFF;
	if (pin==SOFTPOT2_PIN && SP_RELEASE[1]==SP_RELEASE_KEEP) SOFTPOT_SHUTUP=OFF;
	
  //---------------------------------------------------------------------------------
	// Etape 2 : La valeur doit-elle tre prise en compte ?
	//
	// Pas prise en compte si :
	//    - Le capteur qui parle est sens tre inactif (filtrage interfrences)
	//    - Le capteur qui parle a mis la mme valeur que prcdemment
  //---------------------------------------------------------------------------------
	
	// Suppression des problmes d'interfrence sur les controleurs  statut (ON/OFF)
	// et sur les pins non connects  un capteur
	filtertest=0;
	
	if ((pin==DBEAM1_PIN && DBEAM1_STATUS==OFF) || (pin==DBEAM2_PIN && DBEAM2_STATUS==OFF) ||
	    (pin==SOFTPOT1_PIN && SOFTPOT1_STATUS==OFF) || (pin==SOFTPOT2_PIN && SOFTPOT2_STATUS==OFF) ||
	    (pin==EXPPED_PIN && EXPPEDAL_ACTIVE==OFF))
		filtertest=1;
		
	// DBEAM*_STATUS : Indique l'tat en temps rel (variables globales pour le filtre)
	// DBEAM_SHUTUP  : Indique si le controleur doit mettre sa valeur de bord (variable locale)
	if (DBEAM_SHUTUP==ON || SOFTPOT_SHUTUP==ON) filtertest=0; // Prise en compte
	
	// Si je suis en SPECIAL_CALL, je prends en compte (forage d'une borne)
	// Sinon (SPECIAL_CALL==0 && BTN_FREEZE_PUSHED==1) c'est que la valeur de borne a dj
	// t transmise, donc je GELE LA PRISE EN COMPTE tant que le bouton Freeze est pouss
	if (SPECIAL_CALL>0)
		filtertest=0;
	else if (BTN_FREEZE_PUSHED==1) 
		filtertest=1;
	
	// Exclure les pins connects  aucun contrleur, exclure si la mmoire est en cours d'criture
	if (pin==11 || pin==13 || pin==15 || MEM_BUSY==1) filtertest=1;
		
  //---------------------------------------------------------------------------------
	// Etape 3 : Si la valeur doit tre prise en compte :
	// 
	// 						- Je l'affiche (LCD + Bargraph)
	//						- J'envoie son controleur MIDI
	//						- J'arrte l'ventuelle animation des leds
  //---------------------------------------------------------------------------------
		
	if (filtertest==0)
	{
		// J'effectue les transformations ventuelles INV et MIN/MAX (scale)
		valeur7bits = TRANSFORM_VALUE(pin, valeur7bits);

		// Si la valeur a effectivement chang ou que c'est un appel spcial ou forc :
		if (valeur7bits != CTL_LAST_VALUE[idx] || SPECIAL_CALL>0 || SOFTPOT_SHUTUP || DBEAM_SHUTUP)
		{
			// Si c'est un controleur diffrent du dernier qui a parl
			if (GLOBAL_LAST_TOUCHED_PIN != pin)
			{
				// J'allume la led des joysticks PS2 si l'un deux parle
				if (pin==J3X_PIN || pin==J3Y_PIN || pin==J4X_PIN || pin==J4Y_PIN)
					MIOS_DOUT_PinSet(LED_PS2,1);
				else if (GLOBAL_LAST_TOUCHED_PIN==J3X_PIN || GLOBAL_LAST_TOUCHED_PIN==J3Y_PIN || GLOBAL_LAST_TOUCHED_PIN==J4X_PIN || GLOBAL_LAST_TOUCHED_PIN==J4Y_PIN)
					MIOS_DOUT_PinSet(LED_PS2,0);
			}
		
			// Je conserve la nouvelle valeur
			CTL_LAST_VALUE[idx]=valeur7bits;
			
			// J'affiche l'inversion ventuelle du controleur
			if (CTL_INV[idx]==ON)
				MIOS_DOUT_PinSet(LED_INV,1); // On allume la led Invert/Reset
			else
				MIOS_DOUT_PinSet(LED_INV,0); // On teind la led Invert/Reset
	

			// Je flushe le buffer MIDI au cas o il reste des envois non effectus
			MIOS_MIDI_TxBufferFlush();

			// J'envoie le controleur MIDI (ou la note, ou note off)
			if (   (pin==SOFTPOT1_PIN && SP_MODE[0]==SP_MODE_NOTE)
				   ||(pin==SOFTPOT2_PIN && SP_MODE[1]==SP_MODE_NOTE))
			{
				if (pin==SOFTPOT1_PIN && SP_LASTNOTE[0]!=-1)
				{
	 				MIOS_MIDI_TxBufferPut(127 + CTL_MIDICH[idx]);  /* Note OFF */
					MIOS_MIDI_TxBufferPut(SP_LASTNOTE[0]);
					MIOS_MIDI_TxBufferPut(0x7f);
				}
				
				if (pin==SOFTPOT2_PIN && SP_LASTNOTE[1]!=-1)
				{
	 				MIOS_MIDI_TxBufferPut(127 + CTL_MIDICH[idx]);  /* Note OFF */
					MIOS_MIDI_TxBufferPut(SP_LASTNOTE[1]);
					MIOS_MIDI_TxBufferPut(0x7f);
				}

				if (SOFTPOT_SHUTUP==OFF)
				{
				  MIOS_MIDI_TxBufferPut(143 + CTL_MIDICH[idx]); 	// Note Event at corresponding channel #

				  if (pin==SOFTPOT1_PIN) 
				  	SP_LASTNOTE[0] = 127-valeur7bits;
				  else
				  	SP_LASTNOTE[1] = 127-valeur7bits;

				  MIOS_MIDI_TxBufferPut(127-valeur7bits/*48*/);  	// Note number (0..127)
				  MIOS_MIDI_TxBufferPut(0x7f); 										// Velocity = (0..127)
				}
				else
				{
				}
			}
			else
			{
				// Pour la pitch wheel :
				// =====================
				// Si le n de controleur qui lui est attribu est 0, elle envoie un "vrai"
				// signal MIDI de Pitch Wheel (au lieu du CC#0).
				// Pour toute autre valeur de n de controleur elle envoie un Control Change
				// comme les autres capteurs, sur ce n de controleur
				if (pin==PITCH_PIN && CTL_CC[idx]==0)
				{
					// Ajustement du calibrage : Pour une position centre, je dois envoyer 2000H soit
					// 1000000 puis 0000000, soit : 64 puis 0
					if (valeur7bits==68 || valeur7bits==69)
						valeur7bits=64;
						
					// Envoi des valeurs
		   		MIOS_MIDI_TxBufferPut(223 + CTL_MIDICH[idx]);  /* 0xE0=224 (Canal MIDI 1) */
			  	MIOS_MIDI_TxBufferPut(0); // Pitch wheel change
			    MIOS_MIDI_TxBufferPut(valeur7bits); // Valeur du pitch
			  }
			  
			  // Pour tout autre controleur :
			  // ============================
			  // J'envoie son message MIDI
			  else
		  	{
		   		MIOS_MIDI_TxBufferPut(175 + CTL_MIDICH[idx]);  /* 0xB0=176 (Canal MIDI 1) */
			  	MIOS_MIDI_TxBufferPut(CTL_CC[idx]); // Numro de controleur
			    MIOS_MIDI_TxBufferPut(valeur7bits); // Valeur du controleur
			  }
			}
						 
			// Si l'conomiseur tait en train de tourner :
			if (SS_LENGTH>0) 
			{
				// On rinitialise les 2 bargraphs aux positions des joysticks centraux
				unsigned char LX = TRANSFORM_VALUE(J1X_PIN, MIOS_AIN_Pin7bitGet(J1X_PIN));
				unsigned char LY = TRANSFORM_VALUE(J1Y_PIN, MIOS_AIN_Pin7bitGet(J1Y_PIN));
				unsigned char RX = TRANSFORM_VALUE(J2X_PIN, MIOS_AIN_Pin7bitGet(J2X_PIN));
				unsigned char RY = TRANSFORM_VALUE(J2Y_PIN, MIOS_AIN_Pin7bitGet(J2Y_PIN));
				
				BG_DisplayLeftJoystick(LX,LY);
				BG_DisplayRightJoystick(RX,RY);
		  }

			// Rinitialisation de l'conomiseur
			SS_DELAY = SCREEN_SAVER_DELAY;
			SS_LENGTH= 0;
	
			// Affichage des bargraphs
			if (pin==J1X_PIN || pin==J3X_PIN)
				BG_DisplayLeftJoystick(valeur7bits,255);
			else if (pin==J1Y_PIN || pin==J3Y_PIN)
				BG_DisplayLeftJoystick(255,valeur7bits);
			else if (pin==J2X_PIN || pin==J4X_PIN)
				BG_DisplayRightJoystick(valeur7bits,255);
			else if (pin==J2Y_PIN || pin==J4Y_PIN)
				BG_DisplayRightJoystick(255,valeur7bits);
			else if (pin==SOFTPOT1_PIN || pin==DBEAM1_PIN || pin==PITCH_PIN || pin==POT_LEFT)
				BG_DisplayLeftJoystick(valeur7bits,valeur7bits);
			else if (pin==SOFTPOT2_PIN || pin==DBEAM2_PIN || pin==MOD_PIN || pin==POT_RIGHT)
				BG_DisplayRightJoystick(valeur7bits,valeur7bits);
			else if (pin==EXPPED_PIN)
			{
				BG_DisplayLeftJoystick(valeur7bits,valeur7bits);
				BG_DisplayRightJoystick(valeur7bits,valeur7bits);
			}
	
			// Gestion des timers de refresh (position joysticks)
			// On met la bonne variable (TIMER_BG_LEFTJOYS ou TIMER_BG_RIGHTJOYS)  zro pour
			// couper le rafraichissement, ou alors  JOYS_REFRESH pour commencer le dcompte
			
			if (IS_LEFT(pin))	 TIMER_BG_LEFTJOYS  = JOYS_REFRESH;
    	if (IS_RIGHT(pin)) TIMER_BG_RIGHTJOYS = JOYS_REFRESH;

			if ( BTN_FREEZE_PUSHED==1 
				   || (pin==DBEAM1_PIN && DBEAM_SHUTUP==OFF) 
				   || (pin==SOFTPOT1_PIN && SOFTPOT_SHUTUP==OFF) )	TIMER_BG_LEFTJOYS=0;

			if ( BTN_FREEZE_PUSHED==1 
				   || (pin==DBEAM2_PIN && DBEAM_SHUTUP==OFF) 
				   || (pin==SOFTPOT2_PIN && SOFTPOT_SHUTUP==OFF) )	TIMER_BG_RIGHTJOYS=0;

			// Pour viter par exemple que l'on relche un softpot avant de relacher
			// le freeze (sinon les joys ne sont pas rafraichis)
			if (SPECIAL_CALL>0 &&
				  BTN_FREEZE_PUSHED==0 && (
				  (pin==DBEAM1_PIN && DBEAM1_STATUS==OFF) ||
				  (pin==SOFTPOT1_PIN && SOFTPOT1_STATUS==OFF))) TIMER_BG_LEFTJOYS=JOYS_REFRESH;

			if (SPECIAL_CALL>0 &&
				  BTN_FREEZE_PUSHED==0 && (
				  (pin==DBEAM2_PIN && DBEAM2_STATUS==OFF) ||
				  (pin==SOFTPOT2_PIN && SOFTPOT2_STATUS==OFF))) TIMER_BG_RIGHTJOYS=JOYS_REFRESH;
						
			// Je mets  jour l'cran dans lequel je me trouve
			DISPLAY_CTL(pin, valeur7bits);
		}
	} 
}
