// Modifiziert und zusammengeführt mit APP_LCD.c/h
//  Michael Sigl

#include <mios32.h>
#include <string.h>
#include "FONT/font.h"


// fixed format, don't change!
#define MIOS32_LCD_FONT_WIDTH_IX    0
#define MIOS32_LCD_FONT_HEIGHT_IX   1
#define MIOS32_LCD_FONT_X0_IX       2
#define MIOS32_LCD_FONT_OFFSET_IX   3
#define MIOS32_LCD_FONT_BITMAP_IX   4


/////////////////////////////////////////////////////////////////////////////
// Global variables
////////////////////////////////////////////////////////////////////////////
u8  LCD_device = 0; // (not done in MIOS32_Init to allow the initialisation of multiple LCDs)
u16 mios32_lcd_column;
u16 mios32_lcd_line;

u16 mios32_lcd_x;
u16 mios32_lcd_y;


/////////////////////////////////////////////////////////////////////////////
// Local variables
/////////////////////////////////////////////////////////////////////////////
typedef struct {
  u8  *memory;
  u16 width;
  u16 height;
  u16 line_offset;
  u8  colour_depth;
} mios32_lcd_bitmap_t;

static mios32_lcd_bitmap_t font_bitmap;
mios32_lcd_bitmap_t MIOS32_LCD_BitmapInit(u8 *memory, u16 width, u16 height, u16 line_offset, u8 colour_depth);
static s32 LCD_BitmapPrint(mios32_lcd_bitmap_t bitmap);






static u8 prev_glcd_selection = 0xfe; // the previous LCD_device, 0xff: all CS were activated, 0xfe: will force the update

u16 mios32_lcd_column;
u16 mios32_lcd_line;

u16 mios32_lcd_x;
u16 mios32_lcd_y;

static void LCD_Data(u8 data);
static void LCD_Cmd(u8 cmd);

static void LCD_GCursorSet(u16 x, u16 y);

static s32 LCD_PrintChar(char c);

static void LCD_Init();







void MIOS32_LCD_Init(){
	
	// disable font bitmap
	font_bitmap.width = 0;
	
	// call application specific init function
	LCD_Init();
	
	// set character and graphical cursor to initial position
	LCD_Cursor(0, 0);
}


void LCD_Cursor(u16 column, u16 line){
	
  // set character position
  mios32_lcd_column = column;
  mios32_lcd_line = line;

  // set graphical cursor depending on font width
  u8 font_width = 6;
  u8 font_height = 8;
  if( font_bitmap.width ) {
    font_width = font_bitmap.width;
    font_height = font_bitmap.height;
  }

  mios32_lcd_x = column * font_width;
  mios32_lcd_y = line * font_height;

  // forward new cursor position to app driver
	LCD_GCursorSet(mios32_lcd_x, mios32_lcd_y);
}




/////////////////////////////////////////////////////////////////////////////
//! Prints a \\0 (zero) terminated string
//! \param[in] str pointer to string
/////////////////////////////////////////////////////////////////////////////
s32 LCD_Schreibe(const char *str){
  
  s32 status = 0;

  while( *str != '\0' )
    status |= LCD_PrintChar(*str++);

  return status;
}


inline s32 LCD_SchreibeFormatiert(const char *format, ...) {
	
	// Lokale Definition der va_list und Makros
	typedef char* va_liste;
	#define va_start(ap, last) (ap = (va_liste)&last + sizeof(last))
	#define va_arg(ap, typ) (*(typ *)((ap += sizeof(typ)) - sizeof(typ)))
	#define va_end(ap) (ap = (va_liste)0)
	
	// Puffer für die formatierte Ausgabe
	char puffer[32]; 
	char *puffer_ptr = puffer; // Zeiger auf die aktuelle Position im Puffer
	const char *index; // Zeiger, um das Format-String zu durchlaufen
	va_liste argumente;
	
	// Initialisiere die Argumentenliste
	va_start(argumente, format);
	
	// Durchlaufe den Format-String und verarbeite jeden Fall
	for(index = format; *index != '\0'; index++){
													// Wenn das aktuelle Zeichen nicht '%' ist, kopiere es direkt in den Puffer
													if (*index != '%') {	*puffer_ptr++ = *index;	continue;	}
													
													// Das nächste Zeichen nach '%' verarbeiten
													index++; // Überspringe das '%' Zeichen
													
													// Verarbeite das nächste Formatzeichen
													switch (*index){
														
																	case 'c':{	// Zeichen-Argument
																				char zeichen = va_arg(argumente, int);
																				*puffer_ptr++ = zeichen;
																				break;
																				}
																				
																	case 'd':{	// Integer-Argument
																				int zahl = va_arg(argumente, int);
																				puffer_ptr += sprintf(puffer_ptr, "%d", zahl);
																				break;
																				}
																				
																	case 's':{	// String-Argument
																				char *string = va_arg(argumente, char *);
																				while (*string != '\0') {	*puffer_ptr++ = *string++;	}
																				break;
																				}
																	default:	// Unbekanntes Formatzeichen, kopiere es direkt in den Puffer
																				*puffer_ptr++ = *index;
																				break;
																	break;
																	}
													}
	
	// Nullterminator zum Ende des Puffers hinzufügen
	*puffer_ptr = '\0';
	
	// Beende die Nutzung der Argumentenliste
	va_end(argumente);
	
	// Schreibe den formatierten Puffer auf das LCD und gib das Ergebnis zurück
	return LCD_Schreibe(puffer);
}





/////////////////////////////////////////////////////////////////////////////
//! Initializes the graphical font<BR>
//! \param[in] *font pointer to font
/////////////////////////////////////////////////////////////////////////////
void LCD_font(u8 *font){
	
  font_bitmap.memory = (u8 *)&font[MIOS32_LCD_FONT_BITMAP_IX] + (size_t)font[MIOS32_LCD_FONT_X0_IX];
  font_bitmap.width = font[MIOS32_LCD_FONT_WIDTH_IX];
  font_bitmap.height = font[MIOS32_LCD_FONT_HEIGHT_IX];
  font_bitmap.line_offset = font[MIOS32_LCD_FONT_OFFSET_IX];
  font_bitmap.colour_depth = 1;
}





/////////////////////////////////////////////////////////////////////////////
//! Prints a single character
//! \param[in] c character to be print
/////////////////////////////////////////////////////////////////////////////
static s32 LCD_PrintChar(char c){
	
	s32 status;
	
	
	if( !font_bitmap.width )
	return -1;    // font not initialized yet!
	
	mios32_lcd_bitmap_t bitmap = font_bitmap;
	bitmap.memory += (bitmap.height>>3) * bitmap.line_offset * (size_t)c;
	status = LCD_BitmapPrint(bitmap);

	// increment cursor
	if( status >= 0 ) {	++mios32_lcd_column;	}

	return status;
}








// APP_LCD /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// pin initialisation >> Additonal CS-Line @ J10B
inline static void LCD_ExtPort_Init(void) {

	int pin;
	for(pin=0; pin<8; ++pin) {	MIOS32_BOARD_J10_PinInit(pin + 8, MIOS32_BOARD_PIN_MODE_OUTPUT_PP); }
}

// set pin directly
inline static void LCD_ExtPort_PinSet(u8 pin, u8 value){	MIOS32_BOARD_J10_PinSet(pin + 8, value);	}



// serial data shift
inline static void LCD_ExtPort_SerDataShift(u8 data, u8 lsb_first) {		// 202 einiges stretches vom orginal entfernt, bei fehlern die vom orginal wieder dazugeben!

  int i;
  if( lsb_first ) {
    for(i=0; i<8; ++i, data >>= 1) {
								      MIOS32_SYS_STM_PINSET(GPIOC, GPIO_Pin_13, data & 1); // J10B.D8 = ser
								      MIOS32_SYS_STM_PINSET_0(GPIOC, GPIO_Pin_14); // J10B.D9 = 0 (Clk)

								      MIOS32_SYS_STM_PINSET_1(GPIOC, GPIO_Pin_14); // J10B.D9 = 1 (Clk)
								    }
  } else {
    for(i=0; i<8; ++i, data <<= 1) {
								      MIOS32_SYS_STM_PINSET(GPIOC, GPIO_Pin_13, data & 0x80); // J10B.D8 = ser
								      MIOS32_SYS_STM_PINSET_0(GPIOC, GPIO_Pin_14); // J10B.D9 = 0 (Clk)

								      MIOS32_SYS_STM_PINSET_1(GPIOC, GPIO_Pin_14); // J10B.D9 = 1 (Clk)
								    }
  }
}




// pulse the RC line after a serial data shift
inline static void LCD_ExtPort_UpdateSRs(void) {

  LCD_ExtPort_PinSet(2, 0); // J10B.D10
  LCD_ExtPort_PinSet(2, 1); // J10B.D10
}



/////////////////////////////////////////////////////////////////////////////
// Initializes the CS pins for GLCDs with serial port
// - 8 CS lines are available at J15
// - additional lines are available J10
/////////////////////////////////////////////////////////////////////////////
static void LCD_SERGLCD_CS_Init(void)	{	LCD_ExtPort_Init();	}





/////////////////////////////////////////////////////////////////////////////
// Sets the CS line of a serial GLCDs depending on LCD_device		202 optimized
// if "all" flag is set, commands are sent to all segments
/////////////////////////////////////////////////////////////////////////////
static void LCD_SERGLCD_CS_Set(u8 value, u8 all){


    // Note: assume that CS lines are low-active!
	if( all ) {
				if( prev_glcd_selection != 0xff ) {
													prev_glcd_selection = 0xff;
													MIOS32_BOARD_J15_DataSet(value ? 0x00 : 0xff);

													// shift data
													int i;
													for(i=2; i>=0; --i) {	LCD_ExtPort_SerDataShift(value ? 0x00 : 0xff, 1);	}

													// update serial shift registers
													LCD_ExtPort_UpdateSRs();
													}
				}
	else{
		if( prev_glcd_selection != LCD_device ) {
													prev_glcd_selection = LCD_device;
													u32 mask = value ? ~(1 << LCD_device) : 0xffffffff;

													MIOS32_BOARD_J15_DataSet(mask);

													int selected_lcd = LCD_device - 8;
													int selected_lcd_sr = selected_lcd / 8;
													u8 selected_lcd_mask = value ? ~(1 << (selected_lcd % 8)) : 0xff;

													// shift data
													int i;
													for(i=2; i>=0; --i) {	u8 data = (i == selected_lcd_sr) ? selected_lcd_mask : 0xff;
																			LCD_ExtPort_SerDataShift(data, 1);
																			}

													// update serial shift registers
													LCD_ExtPort_UpdateSRs();
													}
		}

}



/////////////////////////////////////////////////////////////////////////////
// Initializes application specific LCD driver	202 optimized
/////////////////////////////////////////////////////////////////////////////
void LCD_Init()	{

		MIOS32_BOARD_J15_PortInit(0);

		LCD_SERGLCD_CS_Init();

		// wait 500 mS to ensure that the reset is released
		{
		int i;
		for(i=0; i<500; ++i)
		MIOS32_DELAY_Wait_uS(1000);
		}



		// initialize LCDs
		LCD_Cmd(0xa8); // Set MUX Ratio
		LCD_Cmd(0x3f);

		LCD_Cmd(0xd3); // Set Display Offset
		LCD_Cmd(0x00);

		LCD_Cmd(0x40); // Set Display Start Line

		LCD_Cmd(0xa1); // Set Segment re-map: rotated
		LCD_Cmd(0xc8); // Set COM Output Scan Direction: rotated


		LCD_Cmd(0xda); // Set COM Pins hardware configuration
		LCD_Cmd(0x12);

		LCD_Cmd(0x81); // Set Contrast Control
		LCD_Cmd(0x7f); // middle

		LCD_Cmd(0xa4); // Disable Entiere Display On

		LCD_Cmd(0xa6); // Set Normal Display

		LCD_Cmd(0xd5); // Set OSC Frequency
		LCD_Cmd(0x80);

		LCD_Cmd(0x8d); // Enable charge pump regulator
		LCD_Cmd(0x14);

		LCD_Cmd(0xaf); // Display On

		LCD_Cmd(0x20); // Enable Page mode
		LCD_Cmd(0x02);


		LCD_ExtPort_Init();
}


/////////////////////////////////////////////////////////////////////////////
// Sends data byte to LCD	202	optimized
// IN: data byte in <data>
/////////////////////////////////////////////////////////////////////////////
inline static void LCD_Data(u8 data) {	// 202 war früher nicht inline

		// chip select and DC
		LCD_SERGLCD_CS_Set(1, 0);

		MIOS32_BOARD_J15_RS_Set(1); // RS pin used to control DC

		// send data
		MIOS32_BOARD_J15_SerDataShift(data);


		// increment graphical cursor
		++mios32_lcd_x;

		// if end of display segment reached: set X position of all segments to 0
		if( (mios32_lcd_x % 128) == 0 ) {	LCD_Cmd(0x00); // set X=0
											LCD_Cmd(0x10);	}
}


/////////////////////////////////////////////////////////////////////////////
// Sends command byte to LCD
// IN: command byte in <cmd>
/////////////////////////////////////////////////////////////////////////////
inline static void LCD_Cmd(u8 cmd)	{	// war früher nicht inline

		// select all LCDs
		LCD_SERGLCD_CS_Set(1, 1);

		MIOS32_BOARD_J15_RS_Set(0); // RS pin used to control DC

		MIOS32_BOARD_J15_SerDataShift(cmd);
}


void LCD_Clear_ALL(void)	{

    u8 x, y;

    // use default font
    LCD_font((u8 *)fnt_NORM);

    // send data
    for(y=0; y<8; ++y){
						LCD_Cursor(0, y);

						// select all LCDs
						LCD_SERGLCD_CS_Set(1, 1);

						MIOS32_BOARD_J15_RS_Set(1); // RS pin used to control DC

						for(x=0; x<128; ++x) MIOS32_BOARD_J15_SerDataShift(0x00);
						}
}



void LCD_Clear_NR(u8 nr){

    // use default font
    LCD_font((u8 *)fnt_NORM);

    u8 x, y;

    // send data
    for(y=0; y<8; ++y){
						LCD_Cursor(0, y);

						// select specific LCD
						LCD_SERGLCD_CS_Set(nr, 0);

						MIOS32_BOARD_J15_RS_Set(1); // RS pin used to control DC

						for(x=0; x<128; ++x) MIOS32_BOARD_J15_SerDataShift(0x00);
						}
 }





inline static void LCD_GCursorSet(u16 x, u16 y) {	// war früher nicht inline

    // set X position
    LCD_Cmd(0x00 | (x & 0xf));
    LCD_Cmd(0x10 | ((x>>4) & 0xf));

    // set Y position
    LCD_Cmd(0xb0 | ((y>>3) & 7));
}




/////////////////////////////////////////////////////////////////////////////
// Transfers a Bitmap within given boundaries to the LCD
/////////////////////////////////////////////////////////////////////////////
static s32 LCD_BitmapPrint(mios32_lcd_bitmap_t bitmap) {

	// abort if max. width reached
	if( mios32_lcd_x >= 128 )	{ return -2; }	// LCD width = 128 pixel
	
	// all GLCDs support the same bitmap scrambling
	int line;
	int y_lines = (bitmap.height >> 3);
	
	u16 initial_x = mios32_lcd_x;
	u16 initial_y = mios32_lcd_y;
  
	for(line=0; line<y_lines; ++line){
										// calculate pointer to bitmap line
										u8 *memory_ptr = bitmap.memory + line * bitmap.line_offset;
										
										// set graphical cursor after second line has reached
										if( line > 0 ){
														mios32_lcd_x = initial_x;
														mios32_lcd_y += 8;
														LCD_GCursorSet(mios32_lcd_x, mios32_lcd_y);
														}
										
										// transfer character
										int x;
										for(x=0; x<bitmap.width; ++x)	LCD_Data(*memory_ptr++);
										}

	// fix graphical cursor if more than one line has been print
	if( y_lines >= 1 ){
						mios32_lcd_y = initial_y;
						LCD_GCursorSet(mios32_lcd_x, mios32_lcd_y);
						}
	return 0;
}
